import renpy.store as store
import renpy.exports as renpy
import sys

def callback(event, **kwargs):
    '''
    Makes it so that dialogue plays a 'talking' sound effect
    INPUT: [event] The Ren'py event associated
    [kwargs] (I don't know I'm assuming this is something Ren'py-internal)
    OUTPUT: none
    '''
    if event == "show":
        renpy.music.play("resource/sounds/speech.wav", channel="sound")
    elif event == "slow_done" or event == "end":
        renpy.music.stop(channel="sound")
            
class NPC():
    '''
    The base class for an NPC character. Every instance of this class represents
      a single NPC character
    This defines images, persuasion functions, and stores relationship values.
    '''

    def __init__(self, name, varName, desc="Default NPC description.", 
        align = "none", isInitBlock = False):
        '''
        Constructor.
        Initializes all the variables corresponding to an NPC, as well as
        defining the character-specific images to be used in dialogue scenes.
        
        INPUT: [name] Name of NPC as shown in-game.
        [varName] Name of NPC internally to the game engine. (i.e. without 
          spaces or special characters)
        [desc] A one-line description of the character to be shown in the 
          Planner.
        [align] Political alignment of the NPC ('demo' or 'auth')
        [isInitBlock] DEBUG ONLY; on game runtime, this should be True, which
          will load image files as necessary.  During a run of the test suite,
          this should be False, as the images will already exist and the
          associated statements should not be run
        OUTPUT: none
        '''
        self.__name = name
        self.__variableName = varName
        self.__description = desc
        self.__align = align
        self.__relationship = 0
        self.__reverse = False
        self.__reversedOnce = False
        self.__opponent = "None"
        self.__dialogueColor = (255,0,0,255) # the color of the NPC's dialogue
        self.__persuasionWeights = [0.333,0.333,0.333]
        self.__emotionSprites = ['happy', 'angry', 'normal', 'sad', 'confused',
        'confident', 'tired', 'sweat']
        
        # TODO:make visible to all NPCs and not specific to an instance
        self.MAX_RELATIONSHIP = 5
        self.MIN_RELATIONSHIP = 0

        # defines the images internally in renpy, with format "variableName 
        #  sprite"
        # for example "president happy"
        for sprite in self.__emotionSprites :
            spriteLocation = "resource/%s/%s.png" % (self.__variableName,sprite)
           
            if renpy.exists(spriteLocation) == False :
                sys.stderr.write("Sprite %s does not exist, "+
                "loading default sprite\n" % spriteLocation)
                spriteLocation = "resource/testNPC/%s.png" % (sprite)
            
            if isInitBlock:
                renpy.image((self.__variableName, sprite), spriteLocation)
                if self.__variableName == 'player' :
                    renpy.image(('you', sprite), spriteLocation)
                
        self.__character =  store.Character(self.__name, 
            color=self.__dialogueColor,
            ctc = store.anim.Blink("resource/nextArrow.png"),
            callback=callback)
       
    def doActivity(self, activity):
        '''
        Call this to jump to the right dialogue tree for an activity.
        INPUT: [activity] The activity to perform. Format of ensuing label is 
          %variableName%_activity.
        OUTPUT: calls the appropriate activity dialogue and proceeds from there.
        '''
        label = "%s_%s" % (self.__variableName, activity)
        unknownLabel = "unknown_%s" % activity
        
        if renpy.has_label(label) :
            renpy.jump(label)
        elif renpy.has_label(unknownLabel) :
                renpy.jump(unknownLabel)
        else :      
            renpy.call("unknownActivity")
            
    
    def doRelationship(self):
        '''
        Call this to jump to the relationship dialogue.
        This will call the appropriate dialogue depending on your relationship
          with the NPC. 
        OUTPUT: calls the appropriate relationship dialogue and proceeds from
          there.
        '''   
        label = "unknownRelationship"
        if self.__opponent != "None" and \
                self.getRelationship() > 3 and \
                self.__opponent.getRelationship() > 3 and \
                not self.__reversedOnce:
           # Set flag: You have gotten too friendly with the opponent,
           #  so now it's time to choose who you're siding with!
           self.__reverse = True
           self.__reversedOnce = True
           label = "%s_reverse" % self.__variableName
        elif self.__reverse == True:
           label = "%s_reverseStill" % self.__variableName
        else:
           label = "%s_%s" %(self.__variableName, self.getRelationship())
        
        if renpy.has_label(label):
           renpy.jump(label)
        else:
           # Don't fall off a cliff.
           label = "%s_default" % self.__variableName
           renpy.jump(label)

    # Here are some getters and setters.
    def getVarName(self):
        ''' 
        OUTPUT: The variable name of the NPC. 
        '''
        return self.__variableName

    def getName(self):
        ''' 
        OUTPUT: The name of the NPC. 
        '''
        return self.__name
        
    def getRelationship(self):
        ''' 
        OUTPUT: The current relationship value the player has with the NPC. 
        '''
        return self.__relationship
        
    def getDescription(self):
        ''' 
        OUTPUT: The one-line description of the NPC. 
        '''
        return self.__description
    
    def getAlignment(self):
        ''' 
        OUTPUT: The alignment of the character, "none", "demo" or "auth"
        '''
        return self.__align
    
    def setAlignment(self, align):
        ''' 
        INPUT: The new alignment of the character, "none", "demo" or "auth"
        OUTPUT: None
        '''
        self.__align = align
        
    def getCharacter(self):
        '''
        OUTPUT: The 'speaker' object for the NPC. 
        Use this to actually make the character talk in Ren'py code.
        '''
        return self.__character

    def unReverse(self):
        '''
        When you reconcile your relationship with an NPC, call this
        function!
        '''
        self.__reverse = False
    
    def isReversed(self):
        ''' 
        If a NPC currently dislikes you ("is reversed").
        OUTPUT: True if that is the case.
        '''
        return self.__reverse
    
    def setOpponent(self, char):
        '''
        This sets the 'counterpart' NPC for the current NPC.
        INPUT: An NPC
        '''
        self.__opponent = char
        
    def changeRelationship(self, delta) :
        ''' 
        Changes the relationship value of this NPC by a relative delta value
        INPUT: [delta] The amount to change the relationship by.
        OUTPUT: A list of Ren'py formatted dialogue that describes the change.
        '''
        oldVal = self.__relationship
        
        self.setRelationship(self.__relationship + delta)
 
        change = self.__relationship - oldVal
        dialogue = []
        if change > 0 :
            color = "{color=0f0}"
            ending = "!"
        elif change < 0:
            color = "{color=f00}"
            ending = "..."
        else:
            return ["Your relationship did not change."]
            
        # Display some scene text indicating your new relationship level.
        relationshipAction = { 
            1:"%sYou are now acquaintances with %s%s{/color}",
            2:"%sYou feel like you and %s could be friends soon%s{/color}",
            3:"%sYou are now friends with %s%s{/color}",
            4:"%sYou are now good friends with %s%s{/color}",
            5:"%sYou are now best friends with %s%s{/color}"
        }
        
        dialogueStr = relationshipAction.get(self.__relationship, "")
        
        dialogue.append(dialogueStr % (color, self.__name, ending))
        
        return dialogue    
            
    def setRelationship(self, value) :
        '''
        setRelationship sets the relationship to an absolute value
        INPUT: [value] The value to change the relationship to.
        OUTPUT: none
        '''
        self.__relationship = max(min(value, self.MAX_RELATIONSHIP), 
                                self.MIN_RELATIONSHIP)
    
    def getPersuasionWeights(self):
        '''
        OUTPUT: The list of persuasion weights for this NPC (in the order money,
          charm, intelligence)
        '''
        return self.__persuasionWeights
        
    def setPersuasionWeights(self, money, charm, intel):
        '''
        INPUT: [money, charm, intel] The persuasion weights for money, charm, 
          and intelligence, respectively.
        '''
        if money+charm+intel != 1:
            # Avoid integer division
            money = float(money)
            charm = float(charm)
            intel = float(intel)
            
            tot = money+charm+intel
            money = money/tot
            charm = charm/tot
            intel = intel/tot
        self.__persuasionWeights = [money, charm, intel]
    
    
    def persuade(self, stats, day):
        ''' 
        Checks if a character is 'persuaded' by the player. It should be harder to persuade
        as the game goes on, but easier if their relationship is good.
        Persuading someone depends on your stat totals (and their stat preferences.) 
        INPUT: [stats] A map of the stats of the player character.
        [day] What in-game day it is. 
        OUTPUT: True if the NPC is successfully persuaded, False otherwise. 
        '''
        money = stats['money']
        charm = stats['charm']
        intel = stats['intelligence']
        score = day/4 - self.__relationship
        weights = self.__persuasionWeights
        playerScore = weights[0]*money+weights[1]*charm+weights[2]*intel
        return score < playerScore
