import renpy.store as store
import renpy.exports as renpy
from NPC import *

class NPCController() :
    '''
    A class that controls all the NPCs in the game, making sure they are 
    instantiated only once, and giving access to the proper NPC when requested.
    
    This should exist as a singleton.
    '''

    def __init__(self, isInitBlock):
        '''
        Constructor.
        Initializes all the NPCs in the game, so if you want to add any NPC
        characters, do it here!
        
        INPUT: [isInitBlock] There are operations done only if initialized
          in the game init block vs. debug mode.  See constructor of NPC class
        OUTPUT: none (constructor)
        '''
        self.__npcList = {
        'none'      : NPC("Player", "player", "", "none", isInitBlock),
        'planner'   : NPC("Plannersaurus", "planner", "", "none", isInitBlock),
        'dinoPres'  : NPC("President Rex", "dinoPres", 
                        "Leader of the Dinosaur Nation.", "demo", isInitBlock),
        'leadRobo'  : NPC("Leader-Bot 3.0", "leadRobo", 
                        "Leader of the Robot Nation.", "auth", isInitBlock),
        'pteradude' : NPC("Pteradude", "pteradude", 
                        "A student at the University of Dinology.", "auth",
                        isInitBlock),
        'gramps'    : NPC("Mr. Velocigramps", "gramps", 
                        "An elderly robot-dinosaur.", "demo", isInitBlock)
        }

        self.__npcList['dinoPres'].setOpponent(self.__npcList['leadRobo'])
        self.__npcList['leadRobo'].setOpponent(self.__npcList['dinoPres'])
        self.__npcList['pteradude'].setOpponent(self.__npcList['gramps'])
        self.__npcList['gramps'].setOpponent(self.__npcList['pteradude'])

        self.__npcList['dinoPres'].setPersuasionWeights(2, 3, 1)
        self.__npcList['leadRobo'].setPersuasionWeights(2, 1, 3)
        self.__npcList['pteradude'].setPersuasionWeights(3, 2, 1)
        self.__npcList['gramps'].setPersuasionWeights(1, 3, 2)

    def getAlignScore(self, align):
        '''
        INPUT: [align] The political alignment ('demo' or 'auth')
        OUTPUT: The sum of the player's relationships with charcters of alignment <align>.
        '''
        score = 0
        for npc in self.__npcList.values():
            if npc.getAlignment() == align and not npc.isReversed():
                score = score + npc.getRelationship()
        return score
        
    def getAlignChars(self, align):
        '''
        OUTPUT: A list of the characters of alignment <align>.
        '''
        characters = []
        for npc in self.__npcList.values():
            if npc.getAlignment() == align:
                characters.append(npc)
        return characters
        
    def getAlignNotReverseChars(self, align):
        '''
        INPUT: [align] The political alignment ('demo' or 'auth')
        OUTPUT: A list of the characters of alignment <align> who are not reversed.
        '''
        friends = []
        for npc in self.__npcList.values():
            if npc.getAlignment() == align and not npc.isReversed():
                friends.append(npc)
        return friends
        
    def getAlignFriends(self, align):
        '''
        INPUT: [align] The political alignment ('demo' or 'auth')
        OUTPUT: A list of the characters of alignment <align> with whom the player has a 
        relationship of 4 or more.
        '''
        friends = [] 
        for npc in self.__npcList.values():
            if npc.getAlignment() == align and not npc.isReversed() \
            and npc.getRelationship() >= 4:
                friends.append(npc)
        return friends
        
    def getFinalScore(self, align, stats):
        '''
        INPUT: [align] The political alignment ('demo' or 'auth')
        [stats] Stat list
        OUTPUT: Your final score calculated 
        '''
        score = self.getAlignScore(align)
        for stat in stats.values():
            score = score + stat/5
        return score
    
    def getNPC(self, name) :
        ''' 
        INPUT: [name] The variable name of the NPC desired.
        OUTPUT: The NPC object corresponding to the NPC desired.
        '''
        return self.__npcList[name]
        
    def getAllNames(self):
        ''' 
        OUTPUT: A list of all the variable names of the NPC characters in the game. 
        '''
        return self.__npcList.keys()
                
                

