import renpy.store as store
import renpy.exports as renpy
from PlannerData import *
from Tooltip import *

class Planner():
    ''' 
    The Day Planner menu. Has a main screen that shows player stats and 
    chosen activities, and also has subscreens where they can select 
    these activities.

    In addition, stores the global game state such as what day it is,
    and unlocked activities.
    '''  

    def __init__(self, playerObj):
        '''
        Constructor.
        Initializes the PlannerData helper class also.
        
        INPUT: [playerObj] A reference to the player, so that we can read in their stats.
        OUTPUT: none (constructor)
        '''
        self.myPlayer = playerObj
        self.plannerData = PlannerData()
        
        self.__day = 0
        self.FINAL_DAY = 14
        
        # The choices that you've made in this menu.
        self.noSelection = "none" # The value when there's no selected activity. 
        
        self.initialize()
         
        
    def initialize(self):
        '''
        Called at the beginning of every planning period. Resets the day's plans.
        OUTPUT: none
        '''
        self.__planMorning = self.noSelection
        self.__planAfternoon = self.noSelection
        self.__partnerMorning = self.noSelection
        self.__partnerAfternoon = self.noSelection
        
    def incrementDay(self):
        ''' 
        Called at the start of every day. Increments the day by 1. 
        OUTPUT: none
        '''
        self.__day += 1
        
    def setDay(self, value):
        ''' 
        INPUT: [value] The final value of day
        OUTPUT: none
        '''
        self.__day = value
            
    def getDay(self):
        ''' 
        OUTPUT: The current in-game day. 
        '''
        return self.__day
        
    def isFinalDay(self):
        '''
        Is it the last day of the game? (Or, somehow later)
        OUTPUT: True or False depending on whether you're done.
        '''
        return self.__day >= self.FINAL_DAY

    def getPlanMap(self):
        ''' 
        Returns a dictionary of all the choices made on the planner menu 
            (what activities and people the player chose)
        OUTPUT: A dictionary of the choices made by the player in the planner. 
        '''
        planDict = {
            'planMorning'       : self.__planMorning,
            'planAfternoon'     : self.__planAfternoon,
            'partnerMorning'    : self.__partnerMorning,
            'partnerAfternoon'  : self.__partnerAfternoon
        }
        return planDict                                                    
        
    def unlockNPC(self, npc) :
        ''' 
        Unlocks an NPC for selection in the day planner.
        INPUT: [_npc] The variable name of the NPC to be unlocked. 
        '''
        self.plannerData.unlockNPC(npc)
        
    def lockNPC(self, npc) :
        ''' 
        Locks an NPC, making it unavailable for selection in the day planner.
        INPUT: [_npc] The variable name of the NPC to be locked.
        '''
        self.plannerData.lockNPC(npc)    
        
    def unlockActivity(self, period, activity) :
        ''' 
        Unlocks an activity for selection in the day planner. 
        INPUT: [_period] The period for which to unlock the activity. 
        [_activity] The name of the activity to unlock. 
        '''
        self.plannerData.unlockActivity(period, activity)
        
    def validPlan(self) :
        ''' 
        Checks whether the planning for a day is complete.
        OUTPUT: True if a player can 'Start Day' in the planner, False otherwise.
        '''
        if self.__planMorning == self.noSelection or \
            self.__planAfternoon == self.noSelection :
            return False
        if self.__planMorning in self.plannerData.groupActivities and \
            self.__partnerMorning == self.noSelection :
            return False
        if self.__planAfternoon in self.plannerData.groupActivities and \
            self.__partnerAfternoon == self.noSelection :
            return False 
        return True
        
    def mainmenu(self):
        ''' 
        Displays the main day planner screen. (main screen that shows player 
            stats and chosen activities) 
        Largely uses Ren'Py UI drawing code and element placement
        OUTPUT: The option chosen on the main day planner. 
            (planMorningActivity/Person, planAfternoonActivity/Person) 
        '''
        renpy.choice_for_skipping()
        
        ## Lots of Ren'Py interaction follows
        
        roll_forward = renpy.roll_forward_info()
    
        # Define box for the day
        renpy.ui.hbox(xpos=60, ypos=47, spacing=50, xanchor='left')     
        renpy.ui.text("Day %s Planning - Overview" % self.getDay())
        renpy.ui.close()
        
        # Define frame for stats
        renpy.ui.frame(xpos=60, ypos=140, spacing=100, xanchor='left')
        
        renpy.ui.vbox()
        renpy.ui.text(self.myPlayer.getName())
        
        # Money icon and rating
        renpy.ui.hbox()
        renpy.ui.text("{image=resource/icons/money.png} Money: %s" % 
            (self.myPlayer.getStatAdjective('money')))
        renpy.ui.close()
        
        # Intelligence icon and rating
        renpy.ui.hbox()
        renpy.ui.text("{image=resource/icons/intel.png} Intelligence: %s" % 
            (self.myPlayer.getStatAdjective('intelligence')))
        renpy.ui.close()
        
        # Charm icon and rating
        renpy.ui.hbox()
        renpy.ui.text("{image=resource/icons/charm.png} Charm: %s" % 
            (self.myPlayer.getStatAdjective('charm')))
        renpy.ui.close()
        renpy.ui.close()
        
        # Define box for planning the morning activity
        renpy.ui.hbox(xpos = 68, ypos = 297, spacing=35)            
        renpy.ui.imagebutton(self.plannerData.activityIconPath +
            self.__planMorning + ".png",
            self.plannerData.activityIconPath + self.__planMorning + "H.png", 
            clicked=renpy.ui.returns("planMorningActivity"))
        renpy.ui.text("with", ypos=25)
        if self.__planMorning in self.plannerData.groupActivities :
            renpy.ui.imagebutton(self.plannerData.peopleIconPath + 
                self.__partnerMorning + ".png", 
                self.plannerData.peopleIconPath + self.__partnerMorning + "H.png", 
                clicked=renpy.ui.returns("planMorningPerson"))
        else :
            renpy.ui.imagebutton(self.plannerData.planIconDisabled,
                self.plannerData.planIconDisabled,
                clicked=None)
        renpy.ui.close()
    
        # Define box for planning the afternoon activity
        renpy.ui.hbox(xpos = 68, ypos = 450, spacing=35)                  
        renpy.ui.imagebutton(self.plannerData.activityIconPath + 
            self.__planAfternoon + ".png",
            self.plannerData.activityIconPath + self.__planAfternoon + "H.png",
            clicked=renpy.ui.returns("planAfternoonActivity"))
        renpy.ui.text("with", ypos=25)
        if self.__planAfternoon in self.plannerData.groupActivities :
            renpy.ui.imagebutton(self.plannerData.peopleIconPath + 
                self.__partnerAfternoon + ".png", 
                self.plannerData.peopleIconPath + self.__partnerAfternoon + "H.png", 
                clicked=renpy.ui.returns("planAfternoonPerson"))
        else :
            renpy.ui.imagebutton(self.plannerData.planIconDisabled, 
                self.plannerData.planIconDisabled, 
                clicked=None)
        renpy.ui.close()
        
        # Define box for placing the "Start Day" button
        renpy.ui.hbox(xpos = 660, ypos = 500, spacing=50)               
        if self.validPlan() :
            renpy.ui.imagebutton("resource/plannerMenu/startday.png",
                "resource/plannerMenu/startdayH.png",
                clicked=renpy.ui.returns("done"))
        else :
            renpy.ui.imagebutton("resource/plannerMenu/startdayD.png",
                "resource/plannerMenu/startdayD.png",
                clicked=None)
        renpy.ui.close()
        
        renpy.shown_window()
        rv = renpy.ui.interact(mouse='menu', type='menu', 
            roll_forward=roll_forward)
        renpy.checkpoint(rv)
        return rv
        
    def planActivity(self, timePeriod):
        ''' 
        Displays day planner subscreen where an activity for a given time
            period can be selected. 
        INPUT: [timePeriod] The given time period from which activities can 
            be chosen. 
        OUTPUT: The activity selected by the user.
        '''
        renpy.choice_for_skipping()
        
        ## More Ren'Py interaction
        roll_forward = renpy.roll_forward_info()
        
        # Define box for the day and info
        renpy.ui.hbox(xpos=60, ypos=47, spacing=50, xanchor='left')     
        renpy.ui.text("Day %s Planning - Select %s activity" % 
            (self.getDay(), timePeriod))
        renpy.ui.close()
        
        # Define grid for selecting activities
        renpy.ui.window(style=store.style.menu_window, xanchor='left', xpos=0.1)
        renpy.ui.grid(3,2)
        for act in self.plannerData.allActivities[timePeriod]:
            if act in self.plannerData.getActivities(timePeriod):
                tt = Tooltip(60, 140, self.plannerData.activityDescription[act])
                renpy.ui.imagebutton(self.plannerData.activityIconPath + act + ".png", 
                    self.plannerData.activityIconPath + act + "H.png", 
                    clicked=renpy.ui.returns(act),
                    hovered=tt.show, unhovered=tt.hide)
            else:
                renpy.ui.imagebutton(self.plannerData.planIconDisabled,
                    self.plannerData.planIconDisabled, clicked=None)
        renpy.ui.close()
        
        renpy.shown_window()
        rv = renpy.ui.interact(mouse='menu', type='menu', roll_forward=roll_forward)
        renpy.checkpoint(rv)
        
        if timePeriod == 'morning' :
            self.__planMorning = rv
            self.__partnerMorning = self.noSelection
        elif timePeriod == 'afternoon' :
            self.__planAfternoon = rv
            self.__partnerAfternoon = self.noSelection
                            
    def planPerson(self, selectedActivity):
        ''' 
        Displays day planner subscreen where the person to interact with for
            a specific activity can be selected. 
        INPUT: [_selectedActivity] The name of the chosen activity. 
        OUTPUT: The person selected by the user.
        '''
        renpy.choice_for_skipping()
        
        ## Yup
        roll_forward = renpy.roll_forward_info()
        
        # Define box to for day and info
        renpy.ui.hbox(xpos=60, ypos=47, spacing=50, xanchor='left')      
        renpy.ui.text("Day %s Planning - Select who to %s with" %
            (self.getDay(), selectedActivity))
        renpy.ui.close()
        
        # Define grid for selecting NPCs
        renpy.ui.window(style=store.style.menu_window, xanchor='left', xpos=0.1)
        renpy.ui.grid(2,2)
        for per in self.plannerData.allNPCs:
            if per in self.plannerData.getNPCs():
                thisNPC = store.getNPC(per);
                npcName = thisNPC.getName()
                relationshipAdj = self.plannerData.getRelationshipAdj()
                tt = Tooltip(60, 140, "%s\nRelationship with %s: %s" %
                    (thisNPC.getDescription(), npcName,
                    relationshipAdj[thisNPC.getRelationship()] ))
                renpy.ui.imagebutton(self.plannerData.peopleIconPath + per + ".png",
                    self.plannerData.peopleIconPath + per + "H.png",
                    clicked=renpy.ui.returns(per),
                    hovered=tt.show, unhovered=tt.hide)
            else:
                renpy.ui.imagebutton(self.plannerData.planIconDisabled, 
                    self.plannerData.planIconDisabled, clicked=None)
        renpy.ui.close()
        
        renpy.shown_window()
        rv = renpy.ui.interact(mouse='menu', type='menu',
            roll_forward=roll_forward)
        renpy.checkpoint(rv)
        
        if selectedActivity in self.plannerData.allActivities['morning'] :
            self.__partnerMorning = rv
        else :
           self.__partnerAfternoon = rv
