import renpy.store as store
import renpy.exports as renpy

class Player():
    '''The player class; contains the player's stats and data
    Accessible in the main game through the variable myPlayer (defined in GameInit.rpy)'''
    
    def __init__(self):
        '''
        Constructor.
        Initializes all player stats to their default starting values.
        
        OUTPUT: none (constructor)
        '''
        self.__player_name = "Player Name"
   
        self.__stats = {
            'money'         : 2,
            'charm'         : 2,
            'intelligence'  : 2
        }
        
        ###
        # Things that aren't likely to change
        ###
        self.MAX_STAT = 20
        self.MIN_STAT = 0
        
        # thresholdLevels are the stat values at which the descriptive adjective
        #   changes
        self.__thresholdLevels = []
        self.__thresholdLevels = [0, 2, 4, 6, 8, 10, 12, 14, 16, 18, 20]
        # threholdNames are the adjectives which describe each level of a stat
        self.__thresholdNames = {
        'money'         : ['Thrifty', 'Survivable', 'Average', 'Decent',
                        'Well-Off', 'Affluent', 'Wealthy', 'Extremely Rich',
                        'Millionaire', 'Billionaire', 'Trillionaire'],
        'charm'         : ['Rough', 'Plain', 'Average', 'Decent', 'Impressive', 
                        'Smooth', 'Popular', 'Charismatic', 'Jaw-Dropping',
                        'Incredible', 'Superstar'],
        'intelligence'  : ['Slacker', 'Average', 'Decent', 'Studious', 
                        'Knowledgable', 'Clever', 'Bright', 'Smart', 
                        'Brilliant', 'Genius', 'Prodigy']
        }
        
    def setName(self, name):
        ''' 
        INPUT: [_name] The player's new name. 
        OUTPUT: none'''
        self.__player_name = name
        
    def getName(self):
        '''            
        OUTPUT: The player's name. 
        '''
        return self.__player_name
    
    def changeStats(self, stat, delta):
        '''
        Modifies the given stat by the respective amount
        And returns a list of dialogue to display for the stat change. 
        INPUT: [stat] The name of the stat to change.
        [delta] The amount to change the stat by.
        OUTPUT: A list of Ren'py formatted dialogue that describes the change.
        '''
        origAdjective = self.getStatAdjective(stat)
        oldVal = self.getStats()[stat]
        
        self.setStats(stat, self.getStats()[stat] + delta)
            
        newAdjective = self.getStatAdjective(stat)  
        change = self.getStats()[stat] - oldVal
            
        dialogue = []
        statChange = {
            4:"{color=0f0}Your %s increased amazingly!{/color}",
            3:"{color=0f0}Your %s increased greatly!{/color}",
            2:"{color=0f0}Your %s increased quite a bit!{/color}",
            1:"{color=0f0}Your %s increased!{/color}",
            -1:"{color=f00}Your %s decreased...{/color}",
            -2:"{color=f00}Your %s decreased quite a bit..{/color}",
            -3:"{color=f00}Your %s decreased greatly...{/color}",
            -4:"{color=f00}Your %s decreased catastrophically...{/color}"
        }
        
        # If it changed at all...
        if change != 0:
            dialogue.append(statChange.get(change,"") % (stat))
        
        # If you reached a new threshold
        if origAdjective != newAdjective :
            if change > 0 :
                dialogue.append("{color=0f0}Your %s is now %s!{/color}" %
                    (stat, newAdjective))
            elif change < 0:
                dialogue.append("{color=f00}Your %s is now %s...{/color}" % 
                    (stat, newAdjective))  
                   
        # If you reach the max, let the player know
        if self.getStats()[stat] == self.MAX_STAT:
            dialogue.append(
            "{color=0f0}Your %s is at maximum!{/color}" % (stat))
        
        return dialogue
        
    def getStatAdjective(self, stat):
        '''
        Gets the descriptive adjective for a specific stat; see the thresholdLevel/Names variables up top
        For example, the range 0-10 has a specific adjective for each stat ('slacker', 'poor', etc).
        INPUT: [stat] The name of the stat to get the adjective for.
        OUTPUT: The descriptive adjective for your current level of that stat.
        '''    
        myLevel = 0
        for i in range(len(self.__thresholdLevels)) :
            if self.__stats[stat] >= self.__thresholdLevels[i] :
                myLevel = i
                
        return self.__thresholdNames[stat][myLevel]        
 
    def setStats(self, stat, delta):
        ''' 
        Sets a stat directly to the passed in value.
        INPUT: [stat] The name of the stat to change.
        [delta] The value to change the stat to.
        OUTPUT: none
        '''
        self.__stats[stat] = max(min(delta, self.MAX_STAT), self.MIN_STAT)
            
    def getStats(self):
        ''' 
        Returns the player's stats.
        OUTPUT: A map of the player's stats.
        '''
        return self.__stats
