#ifndef lint
static char Rcs_Id[] =
    "$Id: fchart.c,v 1.5 2010-04-14 19:24:12-07 geoff Exp $";
#endif

/* GKrellM (C) 1999-2000 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.
|
|  To get a copy of the GNU General Puplic License,  write to the
|  Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/* File Charting plugin for GKrellM
|
|  Copyright 2005, Geoff Kuenning, Claremont, CA, USA
|
|  Author:  Geoff Kuenning <geoff@cs.hmc.edu>
|
|  Version: for gkrellm version 2.2.7 and up
|
|  See README for details
*/

/*
 * $Log: fchart.c,v $
 * Revision 1.5  2010-04-14 19:24:12-07  geoff
 * Close all extra open fds when forking and running external commands.
 *
 * Revision 1.4  2006-03-24 14:05:23-08  geoff
 * Put the version in the title.  Fix the copyright date.  Fix a serious
 * problem that caused segfaults whenever you entered a new item.
 *
 * Revision 1.3  2006/03/16 08:13:29  geoff
 * Add the %w field to the update command.  Add the option to show the
 * first or last N values if the update command produces excess data.
 * Restart udpate commands if the chart width changes.  Reap defunct
 * update processes after killing them.
 *
 * Revision 1.2  2006/03/14 08:48:14  geoff
 * Add the update-on-changes button.  Allow multiple X coordinates in a
 * file.  Support more charted files.
 *
 * Revision 1.1  2006/03/12 01:24:56  geoff
 * Initial revision
 *
 */

#include <assert.h>
#include <ctype.h>
#include <errno.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <gkrellm2/gkrellm.h>


static char *plugin_info[] =
{
N_("File charting plugin for GKrellM, version 1.0.1\n"),
N_("Copyright 2006, Geoff Kuenning, Claremont, CA, USA\n"),
N_("geoff@cs.hmc.edu\n"),
N_("Released under GPL\n"),
N_("\n"),
N_("This plugin updates a chart based on the changing contents of a file.\n"),
N_("Up to three curves can be charted.  The file begins with one or more\n"),
N_("lines containing values to be displayed on the chart; each line gives\n"),
N_("up to three Y values, separated by whitespace.  Excess values are\n"),
N_("ignored.  If multiple lines appear, multiple points will be added to\n"),
N_("the chart in order, so that the first line will appear to the left on\n"),
N_("the chart.\n"),
N_("\n"),
N_("The list of values is terminated by EOF or by a line containing an\n"),
N_("alert code, which may be NORMAL, WARNING, or ALERT.  Following that is\n"),
N_("up to ten strings, each on a separate line.  A multiple-line tooltip\n"),
N_("can also be given, introduced by a line containing \"!!TOOLTIP!!\".\n"),
N_("Finally, the file is terminated by either EOF or by a line containing\n"),
N_("\"!!EOF!!\" (the latter is more reliable in avoiding interesting race\n"),
N_("conditions).  See the README for more information.  The monitoring is\n"),
N_("done at a user-configurable interval.  I suggest that each update be\n"),
N_("done in a single Unix write to avoid races.\n"),
N_("\n"),
N_("The plugin can monitor multiple files.  For each file, you can specify\n"),
N_("the following:\n"),
N_("<i>Label"), N_(" - label of gkrellm chart\n"),
N_("<i>File to chart"), N_(" - file to monitor.\n"),
N_("<i>Update program"), N_(" - a shell command that, when called, will\n"),
N_("continuously update the monitored file.  This program is spawned when\n"),
N_("the chart is first created.  The string \"%w\" is replaced by the\n"),
N_("chart width, and \"%%\" becomes \"%\".\n"),
N_("<i>Warning command"), N_(" - a shell command to run when the state changes\n"),
N_("from normal to warning.\n"),
N_("<i>Alert command"), N_(" - a shell command to run when the state changes\n"),
N_("from normal or warning to alert.\n"),
N_("<i>Check interval"), N_(" - how often to check the file for updates.\n"),
N_("<i>Label format"), N_(" - how to format text on the chart.  Values $0\n"),
N_("through $9 are replaced by the strings from the monitored file; $M is\n"),
N_("the maximum chart value.\n"),
N_("<i>Update only on changes"), N_(" - if this option is checked, the\n"),
N_("chart will be updated only if the file's modify time has changed.\n"),
N_("<i>Show first/last N values"), N_(" - if more values are given than the\n"),
N_("chart can support, show only the first/last N of them.\n"),
N_("\n"),
N_("<b>CREDITS\n"),
N_("The plugin is based on the fmonitor plugin by Anatoly Asviyan.\n"),
N_("\n")
};


#define FC_CONFIG_NAME	"FChart"	/* Name in the configuration window */
#define FC_STYLE_NAME	"fchart"	/* Theme subdir name and gkrellmrc */
#define FC_CONFIG_KEYWORD "FChart"	/* Name in the configuration file */

/*
 * Hardwired implementation limits
 */
#define MAX_CHARTED_FILES	20	/* Limit on number of charted files */
#define MAX_CHARTED_VALUES	3	/* Values charted per file. */
					/* ..WARNING: if you change this, */
					/* ..you must change the call to */
					/* ..gkrellm_store_chartdata in */
					/* ..update_plugin.  Also please */
					/* ..change plugin_info and the */
					/* .. README to match. */
#define MAX_LABEL_LENGTH	15	/* Maximum characters in chart label */
#define MAX_STRING_VALUES	10	/* Max strings displayed per file */
#define FILE_BUFFER_SIZE	8192	/* Bytes to read from file at once */

#define	MIN_GRID_RES    2
#define	MAX_GRID_RES    2000000000

/*
 * Order of fields in the configuration list.  Unfortunately, the
 * structure of gtk pretty much forces us to establish an ordering
 * correspondence between the configuration list and several arrays.
 * It also forces us to keep the entire configuration list in text
 * format, converting certain fields back and forth between numeric
 * and text values.
 *
 * These symbols are used to pull information from the configuration
 * so we can insert it into the fileParams array.
 */
enum {
    FC_LABEL,
    FC_FILE,
    FC_UPDATE,
    FC_WARN,
    FC_ALERT,
    FC_INTERVAL,
    FC_FORMAT,
    FC_UPDATE_ON_CHANGES,
    FC_FIRST_N,
    FC_LAST_N,
    FC_ENTRY_NUM
};

/*
 * Alert/warning states
 */
typedef enum {
    NORMAL, WARNING, ALERT
}
fc_state;

/*
 * Parameters and live data for a single charted file.
 */
typedef struct {
    /*
     * Configuration parameters set by the user
     */
    gchar		*text[FC_ENTRY_NUM]; /* Raw text array from user */
    gchar		*label;		/* Label to give to this chart */
    gchar		*fileName;	/* Name of file being monitored */
    gchar		*update;	/* Command that updates continuously */
    gchar		*warn;		/* Command to run on warnings */
    gchar		*alert;		/* Command to run on alerts */
    gint		interval;	/* Update interval (ticks) */
    gchar		*formatString;	/* Format string for chart labels */
    gint		updateOnChanges; /* True to update only on changes */
    gint		showFirstN;	/* Show first N values, not last N */
    gint		hideExtra;	/* True to hide extra info */
    GkrellmChartconfig	*chartConfig;	/* Configuration of our chart */
    /*
     * Activity variables
     */
    FILE		*file;		/* Access to the file */
    time_t		lastModifyTime;	/* Time file was last modified */
    gint		errorReported;	/* TRUE if file error already told */
    gint		update_pid;	/* PID of update command */
    fc_state		state;		/* Last known alert state */
    GtkWidget		*vbox;		/* Vbox holding the chart */
    GkrellmChart	*chart;		/* Chart we are displaying on */
    GkrellmChartdata	*chartdata[MAX_CHARTED_VALUES];
					/* Record of data being charted */
    gchar		*strings[MAX_STRING_VALUES];
					/* Extra strings to display */
    GkrellmDecal	*stringDecals[MAX_STRING_VALUES];
					/* Decals with strings to display */
    gchar		*tooltipText;	/* Tooltip to show */
    GtkTooltips		*tooltip;	/* Tooltip to show */
    gint		nLiveStrings;	/* Number of live strings found */
} fc_data;

/*
 * File markers
 */
#define FILE_EOF	"!!EOF!!\n"
#define FILE_TOOLTIP	"!!TOOLTIP!!\n"

/*
 * Default label format to use on charts
 */
#define DEFAULT_FORMAT	"$0\\r$1\\b$2\\r$3"

/*
 * Macro for comparing against length-limited constant string
 *
 * Usage:
 *
 *	if (STRING_MATCHES(a, "foo")) ...
 */
#define STRING_MATCHES(a, b) \
			(strncmp(a, b, sizeof b - 1) == 0)

static GkrellmMonitor	*monitor;	/* Pointer to our plugin description */
static GkrellmTicks	*ticker;	/* Internal clock */

static fc_data	fileParams[MAX_CHARTED_FILES];
					/* File parameters */

static int	numChartedFiles = 0;	/* No. of files being monitored */
static int	selectedRow = -1;	/* Row selected in config panel */
					/* ..-1 means none selected */

static int	numConfiguredFiles = 0; /* No. of files currently in the */
					/* ..configuration panel.  During */
					/* ..configuration, this is */
					/* ..different from the number of */
					/* ..monitored files. */

static gint	style_id;

static GtkWidget *fc_vbox;		/* Vbox that all charts are in */
static GtkWidget *entry[FC_ENTRY_NUM];	/* Entries in the config list */
static GtkWidget *config_list;		/* Configuration list */
static GtkWidget *btn_enter, *btn_del;	/* Buttons on the config panel */
static GtkWidget *update_on_changes_button; /* Button on the config panel */
static GtkWidget *first_n_button;	/* Button on the config panel */
static GtkWidget *last_n_button;	/* Button on the config panel */

/*
 * firstN appears twice because it's a radio button.
 */
static char *configName[FC_ENTRY_NUM] = {
    "label", "file", "update", "warn", "alert", "interval", "format",
    "updateOnChanges", "firstN", "firstN",
};

#define EXTRA_NAME	"hide_extra"

static gchar *configTitles[FC_ENTRY_NUM] = {
    _("Label"),
    _("File to Chart"),
    _("Update Command"),
    _("Warning Command"),
    _("Alert Command"),
    _("Charting Interval"),
    _("Label Format"),
    _("Update only on changes"),
    _("Show first N values"),
    _("Show last N values"),
};

#define SAFE_FREE(x) do { if (x != NULL) {g_free(x); (x) = NULL;} } while (0)

static void destroy_fc_panels();
static void create_fc_panels(int first_create);
static void run_update_cmds();
static void refreshChart(fc_data *params);


static gint
chart_expose_event(GtkWidget *widget, GdkEventExpose *ev)
{
    int i;

    for (i = 0; i < numChartedFiles; i++) {
	if (widget == fileParams[i].chart->drawing_area) {
	    gdk_draw_pixmap(widget->window,
		widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		fileParams[i].chart->pixmap, ev->area.x, ev->area.y,
		ev->area.x, ev->area.y,
		ev->area.width, ev->area.height);
	}
    }
    return FALSE;
}

static gint
panel_click(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
    if (event->button == 3)
	gkrellm_open_config_window(monitor);
    return FALSE;
}

/* Shamelessly ripped from the Gtk docs. */
static void fr_message(gchar *message)
{
    GtkWidget *dialog, *label, *okay_button;
    dialog = gtk_dialog_new();
    label = gtk_label_new (message);
    okay_button = gtk_button_new_with_label(_("OK"));
    gtk_signal_connect_object (GTK_OBJECT (okay_button), "clicked",
		GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT(dialog));
    gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->action_area),
		okay_button);
    gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox),
		label);
    gtk_widget_show_all (dialog);
}

/*
 * Parse a buffer containing data to be charted.  The return value is
 * the new state that was stored in the buffer.
 */
static fc_state
parseBuffer(fc_data *params, char *buf, size_t bytesGotten)
{
    size_t	i;			/* Index into buf */
    int		haveTooltip;		/* NZ if tooltip is in buf */
    fc_state	newState;		/* New state that we found */
    char	*stringStart;		/* Beginning of current string */
    gint	values[MAX_CHARTED_VALUES];
					/* Values being charted */
    gint	nLiveValues;		/* Number of live values found */
    gint	nPlotted;		/* Number of x values plotted so far */
    gint	width = gkrellm_chart_width(); /* Width of the chart */

    /*
     * If we got zero bytes, we assume we got stomped on by
     * the update process, so we keep the data from last time.
     */
    if (bytesGotten == 0)
	return params->state;

    /*
     * Make sure the input buffer is null-terminated.  Note that this
     * requires the caller to ensure that the buffer is at least one
     * byte larger than bytesGotten.
     */
    buf[bytesGotten] = '\0';

    /*
     * The buffer consists of up to MAX_CHARTED_VALUES integers on each
     * line.  The data lines are followed by an alert code and then up to
     * MAX_STRING_VALUES strings, optionally followed by an EOF
     * marker.  This isn't easy to parse with sscanf, so we do it all the
     * hard way.  First, parse and plot the data values.
     */
    i = 0;
    nPlotted = 0;
    while (i < bytesGotten) {
	if (buf[i] != '-'  &&  buf[i] != '+' &&  !isdigit(buf[i]))
	    break;

	/* Parse one line */
	nLiveValues = 0;
	values[0] = values[1] = values[2] = 0;
	while (i < bytesGotten) {
	    if (buf[i] == '\n')
		break;
	    else if (!isspace(buf[i])) {
		values[nLiveValues] = atoi(&buf[i]);
		nLiveValues++;
		if (nLiveValues >= MAX_CHARTED_VALUES) {
		    /*
		     * Values array is full; skip rest of this line
		     */
		    while (i < bytesGotten  &&  buf[i] != '\n')
			i++;
		    break;
		}
		while (i < bytesGotten  &&  !isspace(buf[i]))
		    i++;
	    }
	    else
		i++;
	}

	i++;				/* Skip over \n */

	/*
	 * Plot the charted values, but only if either (a) they will
	 * fit within the width of the chart, or (b) we want the last
	 * N values to show.
	 */
	if (!params->showFirstN  ||  nPlotted < width) {
	    /*
	     * WARNING: The following statement is an unfortunate flaw in the
	     * way gkrellm_store_chartdata works.  There is no way to pass a
	     * variable number of arguments to the latter at run time.  So if
	     * you wish to increase MAX_CHART_VALUES, you also have to change
	     * this statement.  Thus the seemingly silly assertion.
	     *
	     * As a side note, the "3" appears as a written-out string in both
	     * the info text and the README.  If you change MAX_CHART_VALUES,
	     * please change those as well.
	     */
	    assert(MAX_CHARTED_VALUES == 3);
	    gkrellm_store_chartdata(params->chart, 0,
	      values[0], values[1], values[2]);
	    gkrellm_draw_chartdata(params->chart);
	}
	nPlotted++;
    }

    /*
     * Parse the alert string.  The alert string is fairly flexible:
     * the word ALERT or WARNING can be followed by any information up
     * to the next newline; anything else is taken as NORMAL.
     * followed immediately by a newline.
     */
    newState = NORMAL;
    if (i < bytesGotten - sizeof "ALERT" + 1
      &&  STRING_MATCHES(&buf[i], "ALERT"))
	newState = ALERT;
    else if (i < bytesGotten - sizeof "WARNING" + 1
      &&  STRING_MATCHES(&buf[i], "WARNING"))
	newState = WARNING;

    /*
     * Skip past the next newline to get to the strings.
     */
    while (i < bytesGotten  &&  buf[i] != '\n')
	i++;

    i++;				/* Skip over \n */
    
    /*
     * Parse the strings.  Note that the STRING_MATCHES calls can
     * potentially overrun the buffer.  This is harmless because it's
     * a stack buffer and we know there's valid data beyond the end.
     * It could be avoided with an extra test against "bytesGotten-i"
     * but it's not worth the trouble.
     */
    params->nLiveStrings = 0;
    stringStart = &buf[i];
    haveTooltip = FALSE;
    while (i < bytesGotten  &&  params->nLiveStrings < MAX_STRING_VALUES) {
	if (STRING_MATCHES(stringStart, FILE_EOF))
	    return newState;
	else if (STRING_MATCHES(stringStart, FILE_TOOLTIP)) {
	    haveTooltip = TRUE;
	    break;
	}

	for (  ;  i < bytesGotten;  i++) {
	    if (buf[i] == '\n') {
		buf[i] = '\0';
		SAFE_FREE(params->strings[params->nLiveStrings]);
		params->strings[params->nLiveStrings] = g_strdup(stringStart);
		params->nLiveStrings++;
		i++;
		break;
	    }
	}

	stringStart = &buf[i];
    }

    /*
     * Parse the tooltip, if it exists.  All we have to do is to skip
     * over the tooltip indicator and set the tooltip to the remainder
     * of the buffer.  However, if the tooltip is empty, we disable it.
     */
    if (params->nLiveStrings >= MAX_STRING_VALUES
      &&  STRING_MATCHES(&buf[i], FILE_TOOLTIP))
	haveTooltip = TRUE;
    if (haveTooltip) {
	i += sizeof FILE_TOOLTIP - 1;
	stringStart = &buf[i];
	if (STRING_MATCHES(&buf[i], FILE_EOF))
	    buf[i] = '\0';
	else {
	    for (  ;  i < bytesGotten;  i++) {
		if (buf[i] == '\n'  &&  i < bytesGotten - sizeof FILE_EOF + 1
		  &&  STRING_MATCHES(&buf[i + 1], FILE_EOF)) {
		    buf[i] = '\0';
		    break;
		}
	    }
	}

	SAFE_FREE(params->tooltipText);
	params->tooltipText = NULL;
	i = strlen(stringStart) - 1;
	if (stringStart[i] == '\n')
	    stringStart[i] = '\0';
	if (stringStart[0] == '\0')
	    gtk_tooltips_disable(params->tooltip);
	else {
	    params->tooltipText = g_strdup(stringStart);
	    gtk_tooltips_set_tip(params->tooltip, params->chart->drawing_area,
	      params->tooltipText, NULL);
	    gtk_tooltips_enable(params->tooltip);
	}
    }

    return newState;
}

static void
update_plugin()
{
    int		i;			/* Handy loop indexes */
    size_t	bytesGotten;		/* Bytes resulting from read */
    char	buf[FILE_BUFFER_SIZE + 1]; /* Buffer for file data */
    fc_state	newState;		/* New GUI state */
    struct stat	fileStat;		/* File status information */

    /*
     * For each monitored file, check to see if it's time to update,
     * and do so if necessary.
     *
     * BUG: the chart is updated at times relative to gkrellm's start
     * time, not relative to when the chart was added.  For long
     * update intervals, this can cause the chart to appear to not
     * work until the first appropriate tick is reached.
     */
    for (i = 0; i < numChartedFiles; i++) {
	fc_data *params = &fileParams[i];
	if (ticker->timer_ticks % params->interval == 0) {
	    /*
	     * Possibly time to update.  Make sure we have the file open
	     */
	    if (params->file == NULL) {
		params->file = fopen(params->fileName, "r");
		if (params->file == NULL) {
		    if (!params->errorReported)
			fprintf(stderr, _("Can't open '%s': %s\n"),
			  params->fileName, strerror(errno));
		    params->errorReported = TRUE;
		    continue;
		}
		params->errorReported = FALSE;
	    }

	    if (fstat(fileno(params->file), &fileStat) == -1)
		fileStat.st_mtime = 0;
	    if (!params->updateOnChanges) 
		params->lastModifyTime = 0;

	    if (fileStat.st_mtime <= params->lastModifyTime)
		newState = params->state;
	    else {
		/*
		 * Read and plot new data.  We count on the fact that
		 * the stdio package buffers data, so as long as the
		 * file is smaller than the buffer size (typically
		 * 8192 bytes), it will be read in a single operation.
		 * This helps to avoid races.
		 */
		rewind(params->file);
		bytesGotten = fread(buf, 1, FILE_BUFFER_SIZE, params->file);
		newState = parseBuffer(params, buf, bytesGotten);
	    }
	    params->lastModifyTime = fileStat.st_mtime;

	    /*
	     * Handle state changes.
	     */
	    if (newState == ALERT  &&  newState != params->state) {
		if (params->alert)
		    system(params->alert);
	    }
	    else if (newState == WARNING  &&  params->state == NORMAL) {
		if (params->warn)
		    system(params->warn);
	    }
	    params->state = newState;

	}
	refreshChart(params);
    }
}

/*
 * Adapted from cpu.c
 */
static void
formatChartData(fc_data *params, gchar *buf, gint size)
{
    gchar c;
    gchar *s;
    gint index;
    gint len;

    if (buf == NULL  ||  size < 1)
	return;
    --size;			/* Make sure there's room for NUL at end */
    *buf = '\0';

    if (params->formatString == NULL)
	return;

    for (s = params->formatString;  *s != '\0'  &&  size > 0;  ++s) {
	len = 1;
	if (*s == '$'  &&  s[1] != '\0') {
	    c = s[1];
	    /*
	     * SEMI-BUG: this code only supports 36 string codes (0-9 and a-z).
	     */
	    if (c == 'M') {
		index = gkrellm_get_chart_scalemax(params->chart);
		len = snprintf(buf, size, "%d", index);
	    }
	    else {
		index = -1;
		if (isdigit(c))
		    index = c - '0';
		else if (islower(c))
		    index = c - 'a';

		if (index >= 0  &&  index < MAX_STRING_VALUES) {
		    if (index >= params->nLiveStrings
		      ||  params->strings[index] == NULL)
			len = 0;
		    else
			len =
			  snprintf(buf, size, "%s", params->strings[index]);
		}
		else {
		    *buf = *s;
		    if (size > 1) {
			*(buf + 1) = *(s + 1);
			++len;
		    }
		}
	    }
	    ++s;
	}
	else
	    *buf = *s;
	size -= len;
	buf += len;
    }
    *buf = '\0';	
}

static void
drawExtra(fc_data *params)
{
    GkrellmChart *cp = params->chart;
    gchar buf[128];

    if (!cp)
	return;
    formatChartData(params, buf, sizeof(buf));
    gkrellm_draw_chart_text(cp, style_id, buf);
}

static void
refreshChart(fc_data *params)
{
    GkrellmChart *cp = params->chart;

    gkrellm_draw_chartdata(cp);
    if (!params->hideExtra)
	drawExtra(params);
    gkrellm_draw_panel_label(cp->panel);
    gkrellm_draw_chart_to_screen(cp);
}

static gint
fc_extra(GtkWidget *widget, GdkEventButton *ev)
{
    int i;

    for (i = 0;  i < numChartedFiles;  i++) {
	if (widget == fileParams[i].chart->drawing_area) {
	    if (ev->type == GDK_BUTTON_PRESS  &&  ev->button == 1) {
		fileParams[i].hideExtra = !fileParams[i].hideExtra;
		refreshChart(&fileParams[i]);
		gkrellm_config_modified();
	    }
	    else if (ev->button == 3
		     || (ev->button == 1 && ev->type == GDK_2BUTTON_PRESS)
		    )
		gkrellm_chartconfig_window_create(fileParams[i].chart);
	    break;
	}
    }

    return FALSE;
}

/*
 * Create the panels for monitored files.
 */
static void
create_plugin(GtkWidget *vbox, gint first_create)
{
    fc_vbox = vbox;
    create_fc_panels(first_create);
}

/*
 * Destroy all existing chart panels
 */
static void
destroy_fc_panels()
{
    int i;

    for (i = 0; i < numChartedFiles; i++) {
	if (fileParams[i].chart) {
	    gkrellm_chart_destroy(fileParams[i].chart);
	    fileParams[i].chart = NULL;
	}
    }
}

static void
create_fc_panels(int first_create)
{
    GkrellmStyle *style;
    int i;
    int j;
    static int last_width = 0;

    style = gkrellm_panel_style(style_id);

    for (i = 0; i < numChartedFiles; i++) {
	fc_data *params = &fileParams[i];

	if (first_create) {
	    /*
	     * Not sure why I need the vbox; copied from cpu.c
	     */
	    params->vbox = gtk_vbox_new(FALSE, 0);
	    gtk_container_add(GTK_CONTAINER(fc_vbox), params->vbox);
	    gtk_widget_show(params->vbox);
	    params->chart = gkrellm_chart_new0();
	    params->chart->panel = gkrellm_panel_new0();
	}

	gkrellm_set_draw_chart_function(params->chart, refreshChart, params);
	gkrellm_chart_create(params->vbox, monitor, params->chart,
	  &params->chartConfig);
	for (j = 0;  j < MAX_CHARTED_VALUES;  j++) {
	    char buf[sizeof "File data XXX"];
	    GkrellmChartdata *cd;

	    snprintf(buf, sizeof buf, "File data %d", j);
	    cd = gkrellm_add_default_chartdata(params->chart, buf);
	    params->chartdata[j] = cd;
	    gkrellm_monotonic_chartdata(cd, FALSE);
	    gkrellm_set_chartdata_draw_style_default(cd, CHARTDATA_LINE);
	    gkrellm_set_chartdata_flags(cd, CHARTDATA_ALLOW_HIDE);
	}

	gkrellm_chartconfig_grid_resolution_adjustment(params->chartConfig,
	  TRUE, 0, (gfloat) MIN_GRID_RES, (gfloat) MAX_GRID_RES, 0, 0, 0, 70);
	gkrellm_chartconfig_grid_resolution_label(params->chartConfig,
	  _("Units drawn on the chart"));

	gkrellm_panel_configure(params->chart->panel, params->label, style);
	gkrellm_panel_create(params->vbox, monitor, params->chart->panel);

        params->tooltip = gtk_tooltips_new();
        gtk_tooltips_disable(params->tooltip);
        params->tooltipText = NULL;
        gtk_tooltips_set_delay(params->tooltip, 1000);

	gkrellm_alloc_chartdata(params->chart);

	if (first_create)
	    {
	    g_signal_connect(G_OBJECT(params->chart->drawing_area),
	      "expose_event", G_CALLBACK(chart_expose_event), params);
	    g_signal_connect(G_OBJECT (params->chart->panel->drawing_area),
	      "expose_event", G_CALLBACK(chart_expose_event), params);

	    g_signal_connect(G_OBJECT(params->chart->drawing_area),
	      "button_press_event", G_CALLBACK(fc_extra), params);
	    g_signal_connect(G_OBJECT(params->chart->panel->drawing_area),
	      "button_press_event", G_CALLBACK(panel_click), params);
	    }
	else
	    refreshChart(params);
    }

    if (first_create  ||  last_width != gkrellm_chart_width())
	run_update_cmds();
    last_width = gkrellm_chart_width();
}

static void
item_sel(GtkWidget *clist, gint row, gint column,
    GdkEventButton *event,
    gpointer data )
{
    gchar *tmp;
    int i;

    selectedRow = row;
    for (i = 0; i < FC_ENTRY_NUM; i++) {
	if (i == FC_UPDATE_ON_CHANGES  ||  i == FC_FIRST_N  ||  i == FC_LAST_N)
	    continue;
	else if (gtk_clist_get_text(GTK_CLIST(config_list), row, i, &tmp)) {
	    gtk_entry_set_text(GTK_ENTRY(entry[i]), tmp);
	}
	else {
	    fprintf(stderr,
	      _("Strange: can't read %d%s col data of selected row %d\n"),
	      i,
	      i == 1 ? _("st") : i == 2 ? _("nd") : i == 3 ? _("rd") : _("th"),
	      row);
	}
    }
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(update_on_changes_button),
      fileParams[selectedRow].updateOnChanges);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(first_n_button),
      fileParams[selectedRow].showFirstN);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(last_n_button),
      !fileParams[selectedRow].showFirstN);
    gtk_widget_set_sensitive(btn_del, TRUE);
    return;
}


static void
item_unsel(GtkWidget *clist, gint row, gint column, GdkEventButton *event,
    gpointer data )
{
    int i;
    char interval[15];

    snprintf(interval, sizeof interval, "%d", gkrellm_update_HZ());

    selectedRow = -1;
    for (i = 0; i < FC_ENTRY_NUM; i++) {
	if (i == FC_INTERVAL)
	    gtk_entry_set_text(GTK_ENTRY(entry[i]), interval);
	else if (i == FC_FORMAT)
	    gtk_entry_set_text(GTK_ENTRY(entry[i]), DEFAULT_FORMAT);
	else if (i == FC_UPDATE_ON_CHANGES)
	    gtk_toggle_button_set_active(
	      GTK_TOGGLE_BUTTON(update_on_changes_button), FALSE);
	else if (i == FC_FIRST_N)
	    gtk_toggle_button_set_active(
	      GTK_TOGGLE_BUTTON(first_n_button), TRUE);
	else if (i == FC_LAST_N)
	    ;
	else
	    gtk_entry_set_text(GTK_ENTRY(entry[i]), "");
    }

    gtk_widget_set_sensitive(btn_del, FALSE);
    return;
}

static void
on_add_click(GtkButton *button, gpointer *data)
{
    char errmsg[80];
    int  i;
    gchar *configData[FC_ENTRY_NUM];

    if (!*gtk_entry_get_text(GTK_ENTRY(entry[FC_FILE]))) {
	fr_message(_("You must specify a file to monitor.\n"));
	return;
    }

    if (selectedRow >= 0) {
	/* We're editing an existing entry */
	gtk_clist_freeze(GTK_CLIST(config_list));
	for (i = 0; i < FC_ENTRY_NUM; i++) {
	    if (i == FC_UPDATE_ON_CHANGES) {
		fileParams[selectedRow].updateOnChanges =
		  gtk_toggle_button_get_active(
		    GTK_TOGGLE_BUTTON(update_on_changes_button));
		gtk_clist_set_text(GTK_CLIST(config_list), selectedRow, i,
		  g_strdup(fileParams[selectedRow].updateOnChanges
		    ? "Y" : "N"));
	    }
	    else if (i == FC_FIRST_N) {
		fileParams[selectedRow].showFirstN =
		  gtk_toggle_button_get_active(
		    GTK_TOGGLE_BUTTON(first_n_button));
		gtk_clist_set_text(GTK_CLIST(config_list), selectedRow, i,
		  g_strdup(fileParams[selectedRow].showFirstN
		    ? "Y" : "N"));
	    }
	    else if (i == FC_LAST_N)
		gtk_clist_set_text(GTK_CLIST(config_list), selectedRow, i,
		  g_strdup(fileParams[selectedRow].showFirstN
		    ? "N" : "Y"));
	    else
		gtk_clist_set_text(GTK_CLIST(config_list), selectedRow, i,
		    gtk_entry_get_text(GTK_ENTRY(entry[i])));
	}
	gtk_clist_thaw( GTK_CLIST(config_list) );
    }
    else {
	/* We're adding a new row */
	if (numConfiguredFiles >= MAX_CHARTED_FILES) {
	    snprintf(errmsg, sizeof errmsg,
	      _("Sorry, but only %d files can be charted.\n"),
	      MAX_CHARTED_FILES);
	    fr_message(errmsg);
	    return;
	}

	for (i = 0; i < FC_ENTRY_NUM; i++) {
	    if (i == FC_UPDATE_ON_CHANGES) {
		fileParams[numConfiguredFiles].updateOnChanges =
		  gtk_toggle_button_get_active(
		    GTK_TOGGLE_BUTTON(update_on_changes_button));
		configData[i] =
		  g_strdup(fileParams[numConfiguredFiles].updateOnChanges
		    ? "Y" : "N");
	    }
	    else if (i == FC_FIRST_N) {
		fileParams[numConfiguredFiles].showFirstN =
		  gtk_toggle_button_get_active(
		    GTK_TOGGLE_BUTTON(first_n_button));
		configData[i] =
		  g_strdup(fileParams[numConfiguredFiles].showFirstN
		    ? "Y" : "N");
	    }
	    else if (i == FC_LAST_N)
		configData[i] =
		  g_strdup(fileParams[numConfiguredFiles].showFirstN
		    ? "N" : "Y");
	    else
		configData[i] =
		  g_strdup(gtk_entry_get_text(GTK_ENTRY(entry[i])));
	}
	numConfiguredFiles++;
	gtk_clist_append (GTK_CLIST(config_list), configData);
	for (i = 0; i < FC_ENTRY_NUM; i++) {
	  g_free(configData[i]);
	}
    }
    return;
}


static void
on_del_click(GtkButton *button, gpointer *data)
{
    if (selectedRow == -1) return;

    gtk_clist_remove(GTK_CLIST(config_list), selectedRow);
    numConfiguredFiles--;
}


static void
create_config_tab(GtkWidget *tab_vbox)
{
    gchar colonTitle[80];
    gchar interval[15];
    GtkWidget *vbox;
    GtkWidget *tabs;
    GtkWidget *text;
    GtkWidget *btn_hbox;
    GtkWidget *edit_table, *scrolled_window;
    GtkWidget *llabel, *lfile, *lupdate, *lwarn, *lalert, *linterval, *lformat;
    GtkWidget *linterval2;

    gint i;

    snprintf(interval, sizeof interval, "%d", gkrellm_update_HZ());

    tabs = gtk_notebook_new();
    gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs), GTK_POS_TOP);
    gtk_box_pack_start(GTK_BOX(tab_vbox), tabs, TRUE, TRUE, 0);

    /* Preferences Tab */
    vbox = gkrellm_gtk_notebook_page(tabs, _("Preferences"));

    // Edit table
    edit_table = gtk_table_new(FC_ENTRY_NUM - 1, 3, FALSE);
    snprintf(colonTitle, sizeof colonTitle, "%s: ", configTitles[FC_LABEL]);
    llabel = gtk_label_new(colonTitle);
    gtk_misc_set_alignment (GTK_MISC(llabel), 1, 1);
    gtk_table_attach(GTK_TABLE(edit_table), llabel, 0, 1, 0, 1, GTK_FILL,
      0, 1, 1);

    entry[FC_LABEL] = gtk_entry_new_with_max_length(MAX_LABEL_LENGTH);
    gtk_table_attach(GTK_TABLE(edit_table), entry[FC_LABEL], 1, 2, 0, 1,
     0, 0, 1, 1);
    llabel = gtk_label_new(" ");
    gtk_misc_set_alignment (GTK_MISC(llabel), 1, 1);
    gtk_table_attach(GTK_TABLE(edit_table), llabel, 2, 3, 0, 1,
      GTK_FILL|GTK_EXPAND, 0, 1, 1);

    snprintf(colonTitle, sizeof colonTitle, "%s: ", configTitles[FC_FILE]);
    lfile = gtk_label_new(colonTitle);
    gtk_misc_set_alignment (GTK_MISC(lfile), 1, 1);
    gtk_table_attach(GTK_TABLE(edit_table), lfile, 0, 1, 1, 2, GTK_FILL,
      0, 1, 1);

    entry[FC_FILE] = gtk_entry_new_with_max_length(255);
    gtk_table_attach(GTK_TABLE(edit_table), entry[FC_FILE], 1, 3, 1, 2,
      GTK_FILL|GTK_EXPAND, 0, 1, 1);

    snprintf(colonTitle, sizeof colonTitle, "%s: ", configTitles[FC_UPDATE]);
    lupdate = gtk_label_new(colonTitle);
    gtk_misc_set_alignment (GTK_MISC(lupdate), 1, 1);
    gtk_table_attach(GTK_TABLE(edit_table), lupdate, 0, 1, 2, 3, GTK_FILL,
      0, 1, 1);

    entry[FC_UPDATE] = gtk_entry_new_with_max_length(255);
    gtk_table_attach(GTK_TABLE(edit_table), entry[FC_UPDATE], 1, 3, 2, 3,
      GTK_FILL, 0, 1, 1);

    snprintf(colonTitle, sizeof colonTitle, "%s: ", configTitles[FC_WARN]);
    lwarn = gtk_label_new(colonTitle);
    gtk_misc_set_alignment (GTK_MISC(lwarn), 1, 1);
    gtk_table_attach(GTK_TABLE(edit_table), lwarn, 0, 1, 3, 4, GTK_FILL,
      0, 1, 1);

    entry[FC_WARN] = gtk_entry_new_with_max_length(255);
    gtk_table_attach(GTK_TABLE(edit_table), entry[FC_WARN], 1, 3, 3, 4,
      GTK_FILL, 0, 1, 1);

    snprintf(colonTitle, sizeof colonTitle, "%s: ", configTitles[FC_ALERT]);
    lalert = gtk_label_new(colonTitle);
    gtk_misc_set_alignment (GTK_MISC(lalert), 1, 1);
    gtk_table_attach(GTK_TABLE(edit_table), lalert, 0, 1, 4, 5, GTK_FILL,
      0, 1, 1);

    entry[FC_ALERT] = gtk_entry_new_with_max_length(255);
    gtk_table_attach(GTK_TABLE(edit_table), entry[FC_ALERT], 1, 3, 4, 5,
      GTK_FILL, 0, 1, 1);

    snprintf(colonTitle, sizeof colonTitle, "%s: ", configTitles[FC_INTERVAL]);
    linterval = gtk_label_new(colonTitle);
    gtk_misc_set_alignment (GTK_MISC(linterval), 1, 1);
    gtk_table_attach(GTK_TABLE(edit_table), linterval, 0, 1, 5, 6, GTK_FILL,
      0, 1, 1);

    entry[FC_INTERVAL] = gtk_entry_new_with_max_length(10);
    gtk_entry_set_text(GTK_ENTRY(entry[FC_INTERVAL]), interval);
    gtk_table_attach(GTK_TABLE(edit_table), entry[FC_INTERVAL], 1, 2, 5, 6,
      GTK_FILL, 0, 1, 1);

    linterval2 = gtk_label_new(_(" gkrellm clock ticks"));
    gtk_misc_set_alignment (GTK_MISC(linterval2), 0, 1);
    gtk_table_attach(GTK_TABLE(edit_table), linterval2, 2, 3, 5, 6, GTK_FILL,
      0, 1, 1);

    snprintf(colonTitle, sizeof colonTitle, "%s: ", configTitles[FC_FORMAT]);
    lformat = gtk_label_new(colonTitle);
    gtk_misc_set_alignment (GTK_MISC(lformat), 1, 1);
    gtk_table_attach(GTK_TABLE(edit_table), lformat, 0, 1, 6, 7, GTK_FILL,
      0, 1, 1);

    entry[FC_FORMAT] = gtk_entry_new_with_max_length(255);
    gtk_entry_set_text(GTK_ENTRY(entry[FC_FORMAT]), DEFAULT_FORMAT);
    gtk_table_attach(GTK_TABLE(edit_table), entry[FC_FORMAT], 1, 3, 6, 7,
      GTK_FILL, 0, 1, 1);

    entry[FC_UPDATE_ON_CHANGES] = gtk_hbox_new(FALSE, 0);
    gkrellm_gtk_check_button(entry[FC_UPDATE_ON_CHANGES],
      &update_on_changes_button, 0, TRUE, 0,
      configTitles[FC_UPDATE_ON_CHANGES]);
    gtk_table_attach(GTK_TABLE(edit_table), entry[FC_UPDATE_ON_CHANGES],
      0, 3, 7, 8, GTK_FILL, 0, 1, 1);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(update_on_changes_button),
      0);

    entry[FC_FIRST_N] = gtk_hbox_new(FALSE, 0);
    first_n_button = gtk_radio_button_new_with_label(NULL,
      configTitles[FC_FIRST_N]);
    gtk_box_pack_start(GTK_BOX(entry[FC_FIRST_N]), first_n_button,
      FALSE, FALSE, 5);
    entry[FC_LAST_N] = NULL;
    last_n_button = gtk_radio_button_new_with_label_from_widget(
      GTK_RADIO_BUTTON(first_n_button), configTitles[FC_LAST_N]);
    gtk_box_pack_start(GTK_BOX(entry[FC_FIRST_N]), last_n_button,
      FALSE, FALSE, 0);
    gtk_table_attach(GTK_TABLE(edit_table), entry[FC_FIRST_N],
      1, 3, 8, 9, GTK_FILL, 0, 1, 1);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(first_n_button), 0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(last_n_button), 0);

    // Buttons to accept/reject config. data in the edit box
    btn_hbox = gtk_hbox_new(FALSE, 5);
    btn_enter = gtk_button_new_with_label(_("Enter"));
    gtk_signal_connect(GTK_OBJECT(btn_enter), "clicked",
		GTK_SIGNAL_FUNC(on_add_click),NULL);
    btn_del = gtk_button_new_with_label(_("Delete"));
    gtk_widget_set_sensitive(btn_del, FALSE);
    gtk_signal_connect(GTK_OBJECT(btn_del), "clicked",
		GTK_SIGNAL_FUNC(on_del_click),NULL);
    gtk_box_pack_start(GTK_BOX(btn_hbox), btn_enter, TRUE, FALSE, 2);
    gtk_box_pack_start(GTK_BOX(btn_hbox), btn_del, TRUE, FALSE, 2);

    // List with saved configurations
    scrolled_window = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window),
      GTK_POLICY_ALWAYS, GTK_POLICY_ALWAYS);

    config_list =
      gtk_clist_new_with_titles(sizeof configTitles / sizeof configTitles[0],
	configTitles);
    gtk_container_add(GTK_CONTAINER(scrolled_window), config_list);
    gtk_signal_connect(GTK_OBJECT(config_list), "select-row",
      GTK_SIGNAL_FUNC(item_sel), NULL);
    gtk_signal_connect(GTK_OBJECT(config_list), "unselect-row",
      GTK_SIGNAL_FUNC(item_unsel), NULL);
    gtk_clist_set_selection_mode(GTK_CLIST(config_list), GTK_SELECTION_SINGLE);

    for (i = 0; i < FC_ENTRY_NUM; i++) {
	int width;
	switch (i) {
	    case FC_LABEL:
		width = 60;
		break;
	    case FC_FILE:
		width = 150;
		break;
	    case FC_INTERVAL:
		width = 60;
		break;
	    case FC_UPDATE_ON_CHANGES:
	    case FC_FIRST_N:
	    case FC_LAST_N:
		width = 10;
		break;
	    default:
		width = 100;
		break;
	}
	gtk_clist_set_column_width(GTK_CLIST(config_list), i, width);
    }

    for (i = 0; i < numChartedFiles; i++) {
	if (fileParams[i].text[FC_INTERVAL] == NULL
	  ||  fileParams[i].interval < 1) {
	    SAFE_FREE(fileParams[i].text[FC_INTERVAL]);
	    fileParams[i].interval = gkrellm_update_HZ();
	    fileParams[i].text[FC_INTERVAL] = g_strdup(interval);
	}
	SAFE_FREE(fileParams[i].text[FC_UPDATE_ON_CHANGES]);
	if (fileParams[i].updateOnChanges)
	    fileParams[i].text[FC_UPDATE_ON_CHANGES] = g_strdup("Y");
	else
	    fileParams[i].text[FC_UPDATE_ON_CHANGES] = g_strdup("N");
	if (fileParams[i].showFirstN) {
	    fileParams[i].text[FC_FIRST_N] = g_strdup("Y");
	    fileParams[i].text[FC_LAST_N] = g_strdup("N");
	}
	else {
	    fileParams[i].text[FC_FIRST_N] = g_strdup("N");
	    fileParams[i].text[FC_LAST_N] = g_strdup("Y");
	}
	if (fileParams[i].text[FC_FORMAT] == NULL) {
	    fileParams[i].text[FC_FORMAT] = g_strdup(DEFAULT_FORMAT);
	    fileParams[i].formatString = fileParams[i].text[FC_FORMAT];
	}
	gtk_clist_append(GTK_CLIST(config_list), fileParams[i].text);
    }

    numConfiguredFiles = numChartedFiles;
    //gtk_clist_columns_autosize(GTK_CLIST(config_list));

    gtk_box_pack_start(GTK_BOX(vbox), edit_table, FALSE, FALSE, 2);
    gtk_box_pack_start(GTK_BOX(vbox), btn_hbox, FALSE, FALSE, 2);
    gtk_box_pack_start(GTK_BOX(vbox), scrolled_window, TRUE, TRUE, 2);

    /* Info Tab */
    vbox = gkrellm_gtk_notebook_page(tabs, _("Info"));
    text = gkrellm_gtk_scrolled_text_view(vbox, NULL,
      GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
    gkrellm_gtk_text_view_append_strings(text, plugin_info,
     sizeof plugin_info / sizeof plugin_info[0]);
}


static void
save_config(FILE *f)
{
    int i, j;
    char chartno[15];

    for (i = 0; i < numChartedFiles; i++) {
	for (j = 0; j < FC_ENTRY_NUM; j++) {
	    if (j != FC_LAST_N)
		fprintf(f, "%s %s:%d:%s\n", FC_CONFIG_KEYWORD, configName[j],
		  i, fileParams[i].text[j] ? fileParams[i].text[j] : "");
	}

	fprintf(f, "%s %s:%d:%d\n", FC_CONFIG_KEYWORD, EXTRA_NAME, i,
	  fileParams[i].hideExtra);

	/*
	 * Save any chart config changes the user has made.
	 */
	snprintf(chartno, sizeof chartno, "%d", i);
	gkrellm_save_chartconfig(f, fileParams[i].chartConfig,
	  FC_CONFIG_KEYWORD, chartno);
    }
}

static void
load_config( gchar* arg )
{
    char *tmp, *tmp2,  *name, *value;
    char buf[15];
    int rowno, i;

    tmp = g_strdup(arg);

    if (strncmp(tmp, GKRELLM_CHARTCONFIG_KEYWORD " ",
	sizeof GKRELLM_CHARTCONFIG_KEYWORD)
       == 0) {
	strtok(tmp, " ");
	name = strtok(NULL, " ");

	rowno = atoi(name);
	if (rowno >= 0  &&  rowno < MAX_CHARTED_FILES) {
	    value = strtok(NULL, "\n");
	    gkrellm_load_chartconfig(&fileParams[rowno].chartConfig, value,
	      MAX_CHARTED_VALUES);

	    if (rowno >= numChartedFiles)
		numChartedFiles = rowno + 1;
	}

	g_free(tmp);

	return;
    }

    name = strtok(tmp, ":");
    if (!name) return;

    tmp2 = strtok(NULL, ":");
    if (!tmp2) return;
    rowno = atoi(tmp2);
    if (rowno < MAX_CHARTED_FILES) {
	value = strtok(NULL, "\n");

	if (strcmp(name, EXTRA_NAME) == 0  &&  value != NULL)
	    fileParams[rowno].hideExtra = atoi(value);
	else {
	    for (i = 0; i < FC_ENTRY_NUM; i++) {
		if (strcmp(name, configName[i]) == 0) {
		    SAFE_FREE(fileParams[rowno].text[i]);
		    if (value)
			fileParams[rowno].text[i] =  g_strdup(value);
		    else
			fileParams[rowno].text[i] =  g_strdup("");
		    break;
		}
	    }
	}

	fileParams[rowno].label = fileParams[rowno].text[FC_LABEL];
	fileParams[rowno].fileName = fileParams[rowno].text[FC_FILE];
	fileParams[rowno].update = fileParams[rowno].text[FC_UPDATE];
	fileParams[rowno].warn = fileParams[rowno].text[FC_WARN];
	fileParams[rowno].alert = fileParams[rowno].text[FC_ALERT];
	if (fileParams[rowno].text[FC_INTERVAL] != NULL)
	    fileParams[rowno].interval =
	      atoi(fileParams[rowno].text[FC_INTERVAL]);
	fileParams[rowno].formatString = fileParams[rowno].text[FC_FORMAT];
	if (fileParams[rowno].text[FC_UPDATE_ON_CHANGES] != NULL) {
	    if (fileParams[rowno].text[FC_UPDATE_ON_CHANGES][0] == 'Y'
	      ||  fileParams[rowno].text[FC_UPDATE_ON_CHANGES][0] == 'y'
	      ||  fileParams[rowno].text[FC_UPDATE_ON_CHANGES][0] == '\0')
		fileParams[rowno].updateOnChanges = 1;
	    else if (fileParams[rowno].text[FC_UPDATE_ON_CHANGES][0] == 'N'
	      ||  fileParams[rowno].text[FC_UPDATE_ON_CHANGES][0] == 'n')
		fileParams[rowno].updateOnChanges = 0;
	    else
		fileParams[rowno].updateOnChanges =
		  atoi(fileParams[rowno].text[FC_UPDATE_ON_CHANGES]);
	}
	if (fileParams[rowno].text[FC_FIRST_N] != NULL) {
	    if (fileParams[rowno].text[FC_FIRST_N][0] == 'Y'
	      ||  fileParams[rowno].text[FC_FIRST_N][0] == 'y'
	      ||  fileParams[rowno].text[FC_FIRST_N][0] == '\0')
		fileParams[rowno].showFirstN = 1;
	    else if (fileParams[rowno].text[FC_FIRST_N][0] == 'N'
	      ||  fileParams[rowno].text[FC_FIRST_N][0] == 'n')
		fileParams[rowno].showFirstN = 0;
	    else
		fileParams[rowno].showFirstN =
		  atoi(fileParams[rowno].text[FC_UPDATE_ON_CHANGES]);
	}

	if (fileParams[rowno].interval < 1) {
	    fileParams[rowno].interval = gkrellm_update_HZ();
	    SAFE_FREE(fileParams[rowno].text[FC_INTERVAL]);
	    snprintf(buf, sizeof buf, "%d", gkrellm_update_HZ());
	    fileParams[rowno].text[FC_INTERVAL] = g_strdup(buf);
	}

	if (rowno >= numChartedFiles)
	    numChartedFiles = rowno + 1;
    }
    g_free(tmp);
}

static void
del_fileParams_entries()
{
    int i, no;
    fc_data *params;

    for (no = 0; no < numChartedFiles; no++) {
	params = &fileParams[no];

	if (params->file != NULL)
	    fclose(params->file);
	params->file = NULL;
	params->errorReported = FALSE;
	params->hideExtra = FALSE;

	for (i = 0; i < FC_ENTRY_NUM; i++) {
	    SAFE_FREE(fileParams[no].text[i]);
	    fileParams[no].text[i] = NULL;
	}
	fileParams[no].label = NULL;
	fileParams[no].fileName = NULL;
	fileParams[no].update = NULL;
	fileParams[no].warn = NULL;
	fileParams[no].alert = NULL;
	fileParams[no].formatString = NULL;
    }
}
  

static void
kill_update_cmds()
{
    int i;

    for (i = 0; i < numChartedFiles; i++) {
	if (fileParams[i].update_pid) {
	    kill(fileParams[i].update_pid, SIGTERM);
	    waitpid(fileParams[i].update_pid, (int *)NULL, 0);
	    fileParams[i].update_pid = 0;
	}
    }
}


static void
run_update_cmds()
{
    int i, pid;
    int j;
    int fd;
    char cmdbuf[BUFSIZ];
    char *cp;

    kill_update_cmds();
    for (i = 0; i < numChartedFiles; i++) {
	if (fileParams[i].update != NULL  &&  *fileParams[i].update != '\0') {
	    pid = fork();
	    if (pid) {
		/* Parent */
		fileParams[i].update_pid = pid;
	    }
	    else {
		j = getdtablesize();
		for (fd = 3;  fd < j;  fd++)
		    close(fd);
		j = 0;
		cp = fileParams[i].update;
		while (j < sizeof cmdbuf - 1 &&  *cp != '\0') {
		    if (*cp == '%') {
			switch (*++cp) {
			    case '\0':
				cmdbuf[j++] = '\0';
				cp--;		/* Re-evaluate \0 */
				break;
			    case '%':
				cmdbuf[j++] = '%';
				break;
			    case 'w':
				/* Insert gkrellm width.  This is a pain. */
				snprintf(&cmdbuf[j], sizeof cmdbuf - j - 1,
				  "%d", gkrellm_chart_width());
				j += strlen(&cmdbuf[j]);
				break;
			    default:
				cmdbuf[j++] = '%';
				cmdbuf[j++] = *cp;
				break;
			}
		    }
		    else
			cmdbuf[j++] = *cp;
		    cp++;
		}
		cmdbuf[j] = '\0';
		execl("/bin/sh", "sh", "-c", cmdbuf, NULL);
		/* Should never return */
		_exit(1);
	    }
	}
    }
}
	 
static void
apply_config()
{
    char *tmp;
    int i;

    selectedRow = -1;

    item_unsel(GTK_WIDGET(config_list), 0, 0, NULL, NULL);
    del_fileParams_entries();
    kill_update_cmds();
    destroy_fc_panels();

    for (numChartedFiles = 0;
      numChartedFiles < MAX_CHARTED_FILES
        &&  gtk_clist_get_text(GTK_CLIST(config_list), numChartedFiles,
	  0, &tmp);
	numChartedFiles++) {

	for (i = 0; i < FC_ENTRY_NUM; i++) {
	    if (gtk_clist_get_text(GTK_CLIST(config_list), numChartedFiles, i,
	      &tmp))
		fileParams[numChartedFiles].text[i] = g_strdup(tmp);
	}

	fileParams[numChartedFiles].label =
	  fileParams[numChartedFiles].text[FC_LABEL];
	fileParams[numChartedFiles].fileName =
	  fileParams[numChartedFiles].text[FC_FILE];
	fileParams[numChartedFiles].update =
	  fileParams[numChartedFiles].text[FC_UPDATE];
	fileParams[numChartedFiles].warn =
	  fileParams[numChartedFiles].text[FC_WARN];
	fileParams[numChartedFiles].alert =
	  fileParams[numChartedFiles].text[FC_ALERT];
	fileParams[numChartedFiles].interval =
	  atoi(fileParams[numChartedFiles].text[FC_INTERVAL]);
	fileParams[numChartedFiles].formatString =
	  fileParams[numChartedFiles].text[FC_FORMAT];

	if (fileParams[numChartedFiles].text[FC_UPDATE_ON_CHANGES][0] == 'Y'
	  ||  fileParams[numChartedFiles].text[FC_UPDATE_ON_CHANGES][0] == 'y'
	  ||  fileParams[numChartedFiles].text[FC_UPDATE_ON_CHANGES][0] == '\0')
	    fileParams[numChartedFiles].updateOnChanges = 1;
	else if (fileParams[numChartedFiles].text[FC_UPDATE_ON_CHANGES][0]
	    == 'N'
	  ||  fileParams[numChartedFiles].text[FC_UPDATE_ON_CHANGES][0] == 'n')
	    fileParams[numChartedFiles].updateOnChanges = 0;
	else
	    fileParams[numChartedFiles].updateOnChanges =
	      atoi(fileParams[numChartedFiles].text[FC_UPDATE_ON_CHANGES]);

	if (fileParams[numChartedFiles].text[FC_FIRST_N][0] == 'Y'
	  ||  fileParams[numChartedFiles].text[FC_FIRST_N][0] == 'y'
	  ||  fileParams[numChartedFiles].text[FC_FIRST_N][0] == '\0')
	    fileParams[numChartedFiles].showFirstN = 1;
	else if (fileParams[numChartedFiles].text[FC_FIRST_N][0]
	    == 'N'
	  ||  fileParams[numChartedFiles].text[FC_FIRST_N][0] == 'n')
	    fileParams[numChartedFiles].showFirstN = 0;
	else
	    fileParams[numChartedFiles].showFirstN =
	      atoi(fileParams[numChartedFiles].text[FC_FIRST_N]);

	if (fileParams[numChartedFiles].file != NULL) {
	    fclose(fileParams[numChartedFiles].file);
	    fileParams[numChartedFiles].file = NULL;
	}
    }

    create_fc_panels(1);
}


#if 0
/*
 * Function to catch all children who die, without risking hanging.
 */
static void my_wait(int signo)
{
    waitpid(-1, NULL, WNOHANG | WUNTRACED);
}
#endif


/* The monitor structure tells GKrellM how to call the plugin routines.
*/
static GkrellmMonitor  plugin_mon =
{
    FC_CONFIG_NAME,     /* Name, for config tab.    */
    0,                  /* Id,  0 if a plugin       */
    create_plugin,      /* The create function      */
    update_plugin,      /* The update function      */
    create_config_tab,  /* The config tab create function   */
    apply_config,       /* Apply the config function        */

    save_config,        /* Save user config     */
    load_config,        /* Load user config     */
    FC_CONFIG_NAME,     /* config keyword       */

    NULL,               /* Undefined 2  */
    NULL,               /* Undefined 1  */
    NULL,               /* private      */

    MON_MAIL,           /* Insert plugin before this monitor            */

    NULL,               /* Handle if a plugin, filled in by GKrellM     */
    NULL                /* path if a plugin, filled in by GKrellM       */
};


/* All GKrellM plugins must have one global routine named init_plugin()
  |  which returns a pointer to a filled in monitor structure.
  */
GkrellmMonitor *
gkrellm_init_plugin()
{

    style_id = gkrellm_add_chart_style(&plugin_mon, FC_STYLE_NAME);
    atexit(kill_update_cmds);
#if 0
    signal(SIGCHLD, my_wait);
#endif
    monitor = &plugin_mon;
    ticker = gkrellm_ticks();
    return &plugin_mon;
}
