/*
 * G    (Graphics)
 *
 * this class holds the graphics code we use
 * in CS 5. It's one letter for ease of typing... .
 */

import java.util.*;
import java.io.File;
import java.awt.*; 
import javax.swing.*;
import java.awt.geom.*;
import java.awt.event.*;
import java.awt.image.*;
import java.awt.font.*;
import javax.imageio.ImageIO;

class GrTimer implements ActionListener
{
	long lastTimeMillis;
	long thisTimeMillis;
	double dt;
	
	javax.swing.Timer timer;
	
	GrTimer()
	{
		timer = new javax.swing.Timer(100,this);
	}
	
	void start() { 
		lastTimeMillis = System.currentTimeMillis();
		timer.start(); 
	}
	void stop() { timer.stop(); }
	void setUpdateTime(int millis) {
		boolean wasRunning = timer.isRunning();
		timer.stop();
		timer = new javax.swing.Timer(millis,this);
		if (wasRunning) { timer.start(); }
	}
	
	public void actionPerformed(ActionEvent e)
	{
		//H.now();
		thisTimeMillis = System.currentTimeMillis();
		dt = (thisTimeMillis - lastTimeMillis)/1000.0; // in seconds
		int end = G.panels.size();
		for (int i=0 ; i<end ; ++i)
		{
			Object o = G.panels.elementAt(i);
			// we update each canvas (window) and let that
			// window update each object the fact that a method
			// named "updateState" is used is not required, but
			// makes sense...
			if (o instanceof GrCanvas)
			{
				((GrCanvas)o).updateState(dt);
			}
		}
	}
}

class G
{
	// get the length of this vector to
	// find the next window's index...
	static Vector panels = new Vector();

	// every 100 milliseconds we update all of the
	// objects held by all of the panels...
	public static GrTimer timer = new GrTimer();
	
	// really returns a panel
	static GrWindow getWindow(int i)
	{
		if (i < panels.size()) return (GrWindow)panels.elementAt(i);
		return null;
	}
	
	static GrControl createControl()
	{
		// used only in the title bar to indicate its index
		int nextWindowIndex = panels.size();
		
		//Suggest that the L&F (rather than the system)
        //decorate all windows.  This must be invoked before
        //creating the JFrame.  Native look and feels will
        //ignore this hint.
		//JFrame.setDefaultLookAndFeelDecorated(true);
		GrWindow appWindow = new GrWindow(nextWindowIndex);

   		// JFrame.EXIT_ON_CLOSE
    	appWindow.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE); 

    	GrControl basePanel = new GrControl();
    	appWindow.getContentPane().add(basePanel, BorderLayout.CENTER);
        appWindow.pack();
        basePanel.requestFocusInWindow();
        appWindow.setVisible(true);
        
        // handle the focus...
        
        // add to the "windows" Vector
        panels.add(basePanel);
		
		return basePanel;
		
	}
	
	// creates a GrWindow and puts it in the "windows" Vector
	static GrCanvas createCanvas()
	{
		// used only in the title bar to indicate its index
		int nextWindowIndex = panels.size();
		
		//Suggest that the L&F (rather than the system)
        //decorate all windows.  This must be invoked before
        //creating the JFrame.  Native look and feels will
        //ignore this hint.
		//JFrame.setDefaultLookAndFeelDecorated(true);
		GrWindow appWindow = new GrWindow(nextWindowIndex);

   		// JFrame.EXIT_ON_CLOSE
    	appWindow.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE); 

    	GrCanvas basePanel = new GrCanvas();
    	appWindow.getContentPane().add(basePanel, BorderLayout.CENTER);
        appWindow.pack();
        basePanel.requestFocusInWindow();
        appWindow.setVisible(true);
        
        // handle the focus...
        
        // add to the "windows" Vector
        panels.add(basePanel);
		
		return basePanel;
	}
}

interface Drawable
{
	// the method called to draw the object
	public void draw(Graphics g);
	
	// the method called every so often to update the object
	// (for example, to animate it)
    public void updateState(double dt);
}

class GrControl extends JPanel //implements ActionListener
{
	// this canvas's drawables
	Vector drawables = new Vector();  // ??
	
	Button b;
	
	// Class, Method, ... to connect to other things...
	
	public GrControl()
	{
		super(); //need a reasonable layout manager here
		b = new Button("Hi");
		// how do you make anonymous classes?
		b.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e)
			{
				H.pl("Wow!");
			}
		});
		add(b);
	}
}

class GrCanvas extends JPanel 
	implements ComponentListener, KeyListener, MouseListener, FocusListener
{
	// this canvas's drawables
	Vector drawables = new Vector();
	
	private int pnlH, pnlW;
	private int coordinateAdjustment;
	private int coordinateDirection;
	private double xmin, xmax, ymin, ymax; // extent of each axis
	private double scalex, scaley;
	private double origX, origY; // the scaled coords of the origin
	
	public GrCanvas()
	{
		super(null); // this indicates _no_ LayoutManager
			
		this.setPreferredSize(new Dimension(500/*width*/,500/*height*/));
		this.setBackground(Color.white);
		this.addKeyListener(this);
		this.addMouseListener(this);
		this.addFocusListener(this);
    	this.addComponentListener(this);
    	
    	
    	// default coordinate system 0-10, 0-10, xy_stretch (keep aspect ratio)
    	xmin = 0;
    	xmax = 10;
    	ymin = 0;
    	ymax = 10;
    	/*
         *     
            public static final int XY_STRETCH_SQUARE_AR = 0;
            public static final int XY_STRETCH_NONSQUARE_AR = 1;
            public static final int XY_ZOOM = 2;
            
            public static final int ORIGIN_UL = 0;
            public static final int ORIGIN_LL = 1;
         *
         */
    	coordinateAdjustment = XY_STRETCH_NONSQUARE_AR; // default Adjustment
    	coordinateDirection = ORIGIN_UL;             // default Direction
    	updateCoords();
	}
	
	public void updateState(double dt)
	{
		int S = drawables.size();
		Object o;
		for (int i=0 ; i<S ; ++i)
		{
			o = drawables.elementAt(i);
			((Drawable)o).updateState(dt);
		}
		repaint();
	}
	
	public void add(Drawable D)
	{
	    // fix coordinates, if necessary
		if (D instanceof GrGrid)
		{
			H.pl("Adding GrGrid");
			GrGrid grg = ((GrGrid)D);
			updateCoords(0,grg.gW,0.0,grg.gH);
		}
		drawables.add(D);

		//repaint(); 
	}
	
	public void remove(Drawable D)
	{
		// run through the drawables and see if one is
		// the same object. If so, remove it.

  		// incredible, Vector provides this functionality
  		// including ignoring the call if the object is absent!
  		
  		drawables.remove(D);
  		//repaint();		
	}
	
	public void paintComponent(Graphics g)
	{
		super.paintComponent(g); // fills in background! important!
		Graphics2D g2 = (Graphics2D)g;
		AffineTransform at = g2.getTransform();
		// if you put a translate here, it moves the axes in the unscaled, pixel units
		g2.scale(scalex,scaley);    // applied in order to the AXES (not the shapes)
		g2.translate(origX,origY);  // note that this moves the axes in the scaled units!
		
		int end = drawables.size();
		
		for (int i=0 ; i<end ; ++i)
		{
			((Drawable)drawables.elementAt(i)).draw(g);
		}
		
		end = drawables.size();
		
		for (int i=0 ; i<end ; ++i)
		{
			((Drawable)drawables.elementAt(i)).draw(g);
		}

		/*
		 * a sine curve
		 */
		 /*
		double xmin = -100;
		double xmax = 100;
		double dx = (xmax-xmin)/1000.0;
		for (double x = xmin+dx ; x<xmax ; x+=dx)
		{
			g2.draw(new Line2D.Double(x-dx,f(x-dx),x,f(x)));
		}
		*/
		g2.setTransform(at); // a matched pair with getTransform above
	}
	
	double f(double x)
	{
		return 200*Math.sin(.1*x);
	}
	
	void updateCoords()
	{
		checkCoords(); // make sure everything's in bounds
		
		// the above call makes sure both of these are strictly positive
		double dx = xmax - xmin;
		double dy = ymax - ymin;
		
		// start everything at 0
		origX = 0;
		origY = 0;
		
		if (coordinateAdjustment == XY_STRETCH_NONSQUARE_AR)
		{
			scalex = pnlW/dx;
			scaley = pnlH/dy;
			
			if (coordinateDirection == ORIGIN_UL)
			{
				origX = -xmin;
				origY = -ymin;
			}
			else /* coordinateDirection == ORIGIN_LL */
			{
				scaley *= -1.0;	
				origX = -xmin;
				origY = -(dy+ymin);
			}
		}
		else if (coordinateAdjustment == XY_STRETCH_SQUARE_AR)
		{
			double extraHeight = 0.0;
			double extraWidth = 0.0;
			// find the largest rectangle of the appropriate
			// aspect ratio within the pnlW, pnlH pixels
			// how to do this? 
			// aspect ratio = height/width
			double ar = ((double)pnlH)/pnlW;
			double desired_ar =  dy/dx;
			if (ar > desired_ar) // too high -- use full width
			{
				//H.pl("Too high");
				scalex = pnlW/dx;
				scaley = scalex;   // need to keep the scales the same
				// how many extra units are there in height?
				extraHeight = pnlH/scaley - dy;
			}
			else // too wide -- use full height
			{
				//H.pl("Too wide");
				scaley = pnlH/dy;
			    scalex = scaley;   // need to keep the scales the same
			    extraWidth = pnlW/scalex - dx;
			}
			
			if (coordinateDirection == ORIGIN_UL)
			{
				origX = -xmin+extraWidth/2;
				origY = -ymin+extraHeight/2;
			}
			else /* coordinateDirection == ORIGIN_LL */
			{
				scaley *= -1.0;	
				origX = -xmin+extraWidth/2;
				origY = -(dy+ymin)-extraHeight/2;
			}
		}
		else
		{
			// ZOOM not yet implemented...
		}
		
		this.repaint();
	}
	
	public void updateCoords(double xmin_, double xmax_, double ymin_, double ymax_)
	{
		this.xmin = xmin_;
		this.xmax = xmax_;
		this.ymin = ymin_;
		this.ymax = ymax_;
		
		// make sure they're legal
		double tmp=0;
		
		updateCoords();  // calls update coords
	}
	
	public void checkCoords()
	{	
		if (xmin == xmax) {
			H.pl("xmin is equal to xmax = " + xmin);
			H.pl("Setting xmin=0, xmax = 10");
			xmin = 0.0; xmax = 10.0;
		}
		if (xmin > xmax) {
			double tmp = xmin;
			xmin = xmax;
			xmax = tmp;
		}
		
		if (ymin == ymax) {
			H.pl("ymin is equal to ymax = " + ymin);
			H.pl("Setting ymin=0, ymax = 10");
			ymin = 0.0; ymax = 10.0;
		}
		if (ymin > ymax) {
			double tmp = ymin;
			ymin = ymax;
			ymax = tmp;
		}
	}
	
	public void setOrigin(int originLocation)
    {
    	if (originLocation == ORIGIN_LL)
    		coordinateDirection = ORIGIN_LL;
    	else	
    	    coordinateDirection = ORIGIN_UL;
    }
	
	public void changeCoordsWith(int changeType)
	{
		// check for legal values
		this.coordinateAdjustment = changeType;
	}
	
	public void componentHidden(ComponentEvent e) {;}

    public void componentMoved(ComponentEvent e) {;}

    public void componentResized(ComponentEvent e) 
    {
    	pnlH = this.getSize().height;
    	pnlW = this.getSize().width;
    	//H.pl("Size = (h,w) = (" + pnlH + "," + pnlW + ")");
    	updateCoords();
    	/* Component c = e.getComponent();
        H.pl("componentResized event from "
          + c.getClass().getName() + "; new size: " + c.getSize().width
          + ", " + c.getSize().height);*/
                       
    }

    public void componentShown(ComponentEvent e) {;}
    public void keyTyped(KeyEvent e)
    {    	
    	char c = e.getKeyChar();
    	//H.pl("key: " + c);
    	
    	if (c == 'x')
    	{
    		H.p("xmin (" + xmin + "): ");
    		xmin = H.nd();
    		updateCoords();
    	}
    	else if (c == 'y')
    	{
    		H.p("ymin: (" + ymin + "): " );
    		ymin = H.nd();
    		updateCoords();
    	}
    	else if (c == 'X')
    	{
    		H.p("xmax (" + xmax + "): ");
    		xmax = H.nd();
    		updateCoords();
    	}
    	else if (c == 'Y')
    	{
    		H.p("ymax: (" + ymax + "): " );
    		ymax = H.nd();
    		updateCoords();
    	}
    	else if (c == 'o')
    	{
    		H.p("origY: (" + origY + "): " );
    		origX = H.nd();
    	}
    	else if (c == 'O')
    	{
    		H.p("origY: (" + origY + "): " );
    		origY = H.nd();
    	}
    	else if (c == 'u')
    	{
    		H.pl("setting coord dir to UL");
    		coordinateDirection = ORIGIN_UL;
    		updateCoords();
    	}
    	else if (c == 'l')
    	{
    		H.pl("setting coord dir to LL");
    		coordinateDirection = ORIGIN_LL;
    		updateCoords();
    	}
    	else if (c == 'q')
    	{
    		System.exit(0);
    	}
    }
    public void keyReleased(KeyEvent e) {;}
    public void keyPressed(KeyEvent e) {;}
    
    public void mouseClicked(MouseEvent e) {;}
    public void mouseEntered(MouseEvent e) {;}
    public void mouseExited(MouseEvent e) {;}
    public void mousePressed(MouseEvent e) 
    {
        if (e.isShiftDown())
        {
            int x = e.getX();
            int y = e.getY();
            // H.pl("pixel coordinates (x,y) are " + x + "," + y);
            AffineTransform at = new AffineTransform();
            at.scale(scalex,scaley);    // stolen from the paintComponent method
            at.translate(origX,origY);  // ditto
            double[] pt = new double[2];
            pt[0] = x; pt[1] = y;
            try {
                at.inverseTransform(pt,0,pt,0,1);
                H.pl("\nPoint clicked:" + 
                     "\n            x: " + H.fmt(pt[0]) +
                     "\n            y: " + H.fmt(pt[1]) + "\n");
            } catch (Exception ex) { 
                ; // ex.printStackTrace(); 
            }
        }
    }
    public void mouseReleased(MouseEvent e) {;}
    
    public void focusGained(FocusEvent e) {;}
    public void focusLost(FocusEvent e) {;}
    
    public static final int XY_STRETCH_SQUARE_AR = 0;
    public static final int XY_STRETCH_NONSQUARE_AR = 1;
    public static final int XY_ZOOM = 2;
    
    public static final int ORIGIN_UL = 0;
    public static final int ORIGIN_LL = 1;

}

class GrWindow extends JFrame
{
	// the index of this window in G.windows
	int windowIndex;
	
	public GrWindow(int index)
	{
		super("Window " + index);
		this.windowIndex = index;
	}
}

class GrImage implements Drawable
{
	private BufferedImage im;
	private double x;
	private double y;
	
	public GrImage(double x, double y, String filename)
	{
		this.x = x;
		this.y = y;
		File f = new File(filename);
		try {
			// GIFs are OK under Win2k...
			// JPGs, too, but not BMPs (Windows Paint converts them)
			im = ImageIO.read(f);
		}
		catch (Exception e)
		{
			H.pl("The image in " + filename + " could not be read!");
			H.pl("No image will be used.\n");
			H.pl("Error details:");
			e.printStackTrace();
			im = null;
		}
	}
	
	public void draw(Graphics g)
	{
		if (im != null) 
		{
			//H.pl("Drawing image");
			Graphics2D g2 = (Graphics2D)g;
    		g2.drawImage(im, new AffineTransform(1f,0f,0f,1f,x,y), null);
    	}
    	else
    	{
    		H.pl("im was null -- the image may not have been found");
    	}
	}
	
	public void updateState(double dt)
	{
		;
	}

}

class GrText implements Drawable
{
    private String text;
    private double x,y;
    private int pts;
    private Color c;
    Shape textPicture;
    
    /*
     *
     *TextLayout textTl = new TextLayout("Text", new Font("Helvetica", 1, 96), new FontRenderContext(null, false, false));
    AffineTransform textAt = new AffineTransform();
    textAt.translate(0, (float)textTl.getBounds().getHeight());
        shapes[2] = textTl.getOutline(textAt);
        */
    
    public GrText(double x, double y, int pts, String text, Color c)
    {
        this.text = text;
        this.x = x;
        this.y = y;
        this.c = c;
        this.pts = pts;
        TextLayout textLO = new TextLayout(text, new Font("Helvetica",1,pts),
                                new FontRenderContext(null,false,false));
        AffineTransform textAt = new AffineTransform();
        textAt.translate(0, (float)textLO.getBounds().getHeight());
        textPicture = textLO.getOutline(textAt);                           
    }
    
    public void draw(Graphics g)
    {
        Graphics2D g2 = (Graphics2D)g;
        g2.setColor(c);
        AffineTransform at = g2.getTransform();
        g2.translate(x,y);
        g2.fill(textPicture);
        g2.setTransform(at);
    }
    
    public void updateState(double dt)
    {
        ;
    }   
}

class GrCircle implements Drawable
{
	private Ellipse2D.Double e;
	private Color c;
	
	public GrCircle(double x, double y, double radius, Color c)
	{
		this.c = c;
		double ulx = x-radius;
		double uly = y-radius;
		e = new Ellipse2D.Double(ulx,uly,2.0*radius,2.0*radius);
	}
	
	public void draw(Graphics g)
	{
		Graphics2D g2 = (Graphics2D)g;
		g2.setColor(c);
		g2.fill(e);
	}
	
	public void updateState(double dt)
	{
		;
	}
}

class GrLine implements Drawable
{
	private Line2D.Double line;
	Color c;
	BasicStroke lineStroke;
	Stroke oldStroke;
	
	public GrLine(double x1, double y1, double x2, double y2, Color c)
	{
		this.c = c;
		line = new Line2D.Double(x1,y1,x2,y2);
		setStrokeWidth(1.0/1000.0); // should be one pixel on most screens
	}
	
	public void setStrokeWidth(double width)
	{
		if (width > 0)
		{
			lineStroke = new BasicStroke((float)width);
		}
		else
		{
			H.pl("Can't make a GrLine stroke with a nonpositive width.");
		}
	}
	
	public void draw(Graphics g)
	{
		Graphics2D g2 = (Graphics2D)g;
		g2.setColor(c);
		if (lineStroke != null)
		{
			oldStroke = g2.getStroke();
			g2.setStroke(lineStroke);
		}
		g2.draw(line);
		if (lineStroke != null)
		{
			g2.setStroke(oldStroke);
		}
	}
	
    public void updateState(double dt)
	{
		;
	}
}

class GrRectangle implements Drawable
{
	private Rectangle2D.Double r;
	private Color c;
	private boolean fill;
	private Rectangle2D.Double r_left;
	private Rectangle2D.Double r_right;
	private Rectangle2D.Double r_top;
	private Rectangle2D.Double r_bottom;
	
	public GrRectangle(Rectangle2D.Double r_in)
	{
		this.r = r_in;
		this.c = Color.blue;
		this.fill = true;
	}
	
	public GrRectangle(double x, double y, double w, double h, Color c)
	{
		this.c = c;
		this.r = new Rectangle2D.Double(x,y,w,h);
		this.fill = true;
	}
	
	public GrRectangle(double x, double y, double w, double h, Color c, boolean fill)
	{
		this.fill = fill;
		this.c = c;
		this.r = new Rectangle2D.Double(x,y,w,h);
		if (!this.fill) {
			r_left = new Rectangle2D.Double(x,y,1,h);
			r_right = new Rectangle2D.Double(x+w-1,y,1,h);
			r_top = new Rectangle2D.Double(x,y,w,1);
			r_bottom = new Rectangle2D.Double(x,y+h-1,w,1);
		}
	}
	
	public void draw(Graphics g)
	{
		Graphics2D g2 = (Graphics2D)g;
		g2.setColor(c);
		if (fill)
			g2.fill(r); 
		else {
			// we draw the four sides...
			g2.fill(r_left);
			g2.fill(r_right);
			g2.fill(r_top);
			g2.fill(r_bottom);
		}
	}
	
	public void updateState(double dt)
	{
		;
	}
}

class GrGrid implements Drawable
{
	protected int gH, gW;   // grid height and width
	protected Color[][] color;  // the grid itself
	
	public GrGrid(int ht, int wd)
	{
		this.gH = ht;
		this.gW = wd;
		color = new Color[gH][gW];
		randomColors();
	}
	
	public void updateState(double dt)
	{
	   ;
	}
	
	public void randomColors()
	{
		for (int r=0 ; r<gH ; ++r) {
			for (int c=0 ; c<gW ; ++c) {
				switch (H.randInt(0,6)) 
				{
				case 1: color[r][c] = Color.magenta; break;
				case 2: color[r][c] = Color.green; break;
				case 3: color[r][c] = Color.red; break;
				case 4: color[r][c] = Color.blue; break;
				case 5: color[r][c] = Color.yellow; break;
				default: color[r][c] = Color.black; break; 
				}
			}
		}
	}
	
	public void draw(Graphics g)
	{
		Graphics2D g2 = (Graphics2D)g;
		for (int r=0 ; r<gH ; ++r) {
			for (int c=0 ; c<gW ; ++c) {
				g2.setColor(color[r][c]);
				g2.fillRect(c,r,1,1); // x,y is c,r
			}
		} 
	}
}

class LifeUpdater
{
    public void updateLife(Color[][] last, Color[][] next)
    {
        // in main:
        //GrCanvas artwork = G.createCanvas();
        //artwork.add(new GrLife(50,50,100,new MyLifeUpdater()));
        //G.timer.start();
        
        // outside of main:
        // class MyLifeUpdater extends LifeUpdater
        // public void updateLife(Color[][] last, Color[][] next)
        // { /* code here for life updating... */ }
        ;
    }
}

class GrCFour extends GrGrid
{
    protected Arc2D.Double[][] checkers;
    Color xcolor, ocolor, ecolor, bg;
    
    public GrCFour(int ht, int wd)
    {
        super(ht,wd);
        initializeCheckers();
    }
    
    public void initializeCheckers()
    {
        checkers = new Arc2D.Double[gH][gW];
        for (int r=0 ; r<gH ; ++r) {
            for (int c=0 ; c<gW ; ++c) {
                checkers[r][c] = new Arc2D.Double(c,r,1,1,0,360,Arc2D.PIE); // x,y is c,r
            }
        } 
        // initialize the game colors -- feel free to change this
        xcolor = Color.black;
        ocolor = Color.red;
        ecolor = Color.white; // empty color
        bg     = Color.yellow; // background color
    }
    
    public GrCFour(char[][] board)
    {
        super(board.length,board[0].length);
        initializeCheckers();
        setBoard(board); // set the colors appropriately
    }
    
    public void setBoard(char[][] board)
    {
        if (board.length != gH || board[0].length != gW)
        {
            H.pl("Error in GrCFour's setBoard method. The graphical");
            H.pl("board is of size h,w = " + gH + "," + gW + " but the");
            H.pl("board that was input to the method is h,w = " + 
                 board.length + "," + board[0].length);
            H.pl("\nIgnoring the setBoard method call.\n");
            return;
        }
        for (int r=0 ; r<gH ; ++r) {
            for (int c=0 ; c<gW ; ++c) {
                switch (board[r][c])
                {
                    case 'x': case 'X': color[r][c] = xcolor; break;
                    case 'o': case 'O': color[r][c] = ocolor; break;
                    default: color[r][c] = ecolor; break;
                }
            }
        }
    }
    
    // overriding GrGrid's draw method
    public void draw(Graphics g)
    {
        //super.draw(g);
        Graphics2D g2 = (Graphics2D)g;
        g2.setBackground(ecolor);
        g2.clearRect(0,0,gW,gH);
        // first, we should draw the background (yellow)
        // then, draw all of the spaces for the checkers (white)
        // then, draw the checkers themselves
        for (int r=0 ; r<gH ; ++r) {
            for (int c=0 ; c<gW ; ++c) {
                g2.setColor(color[r][c]);
                g2.fill(checkers[r][c]);
            }
        } 
    }
}

class GrLife extends GrGrid
{
	Object animator;
	int pauseMillis;
	protected Color[][] colorBuffer; // a second buffer for animating
	private Color[][] colorSwapper;
	
	public GrLife(int ht, int wd, int pauseMillis, Object animator)
	{
		super(ht,wd);
		this.pauseMillis = pauseMillis;
		this.animator = animator;
		colorBuffer = new Color[gH][gW];
		
		// reinitialize colors appropriately, i.e., with white
		// around the edges and only black and white inside...
		for (int r=0 ; r<gH ; ++r) {
			for (int c=0 ; c<gW ; ++c) {
				switch (H.randInt(0,1)) 
				{
				case 1: color[r][c] = Color.white; break;
				case 2: color[r][c] = Color.green; break;
				case 3: color[r][c] = Color.red; break;
				case 4: color[r][c] = Color.blue; break;
				case 5: color[r][c] = Color.yellow; break;
				default: color[r][c] = Color.black; break; 
				}
			}
		}
		for (int r=0 ; r<gH ; ++r) { color[r][0] = color[r][gW-1] = Color.white; }
		for (int c=0 ; c<gW ; ++c) { color[0][c] = color[gH-1][c] = Color.white; }
		for (int r=0 ; r<gH ; ++r) {
			for (int c=0 ; c<gW ; ++c) {
				colorBuffer[r][c] = color[r][c];
			}
		}
	}
	
	public void updateState(double dt)
	{
		// we don't need dt in this case...
		
		// the animator Object has to be of type CS5App to work at all
		if (animator instanceof LifeUpdater)
		{
			((LifeUpdater)animator).updateLife(color,colorBuffer);
			
			// swap the two 2d color arrays
			colorSwapper = color;
			color = colorBuffer;
			colorBuffer = colorSwapper;	
		}
	}
	
	
}