/*
 * mcl.cc
 *
 * Feel free to use the makefile in this directory also.
 *   use 
 *
 *   make mcl
 *
 */

#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <string>
#include <fstream.h>
#include <iostream.h>
#include <stdio.h>
#include <vector>
#include "Timing.hh"
#include "KbdInput.hh"

extern "C" {
#include "Nclient.h"
}

/*
 * global parameters (read from "parameters" file)
 */

double COLOR = 10;               // color of particles before they move
double COLOR2 = 7;               // color of particles after they move
double PARTICLE_RADIUS = 50.0;   // size of drawn particles
double NOISE = 0.0;              // standard deviation of the _actual_
                                 // sensor noise
double NOISE_MODEL = 75.0;       // standard deviation of the _modeled_
                                 // sensor noise
int NPART = 100;                 // number of particles used
double POS_STDEV_FRAC = 0.2;     // fraction of robot moves used as
                                 //   the st. dev. of the distribution
                                 //   on where the robot actually goes

/*
 * symbolic constants
 */

const int POSITIVE_Y = 0;
const int POSITIVE_X = 1;
const int NEGATIVE_Y = 2;
const int NEGATIVE_X = 3;

/*
 * a global timing object and forward declarations
 */

Timing T;
double gaussrand();

/*
 * the Wall class
 */

class Wall
{
  public:
    static const int H = 0;
    static const int V = 1;

    int dir;  // direction
    double x,y,low,high;

    Wall(int dir_, double x_, double y_, double low_, double high_):
      dir(dir_), x(x_), y(y_), low(low_), high(high_) { ; }

    Wall():
      dir(Wall::H), x(.0), y(.0), low(.0), high(.0) { ; }

};

/*
 * the Particle class
 */

class Particle
{
  public:
    double x,y,w; // x, y, and weight

    Particle(double x_, double y_, double w_):
      x(x_), y(y_), w(w_) { ; }

    Particle():
      x(.0), y(.0), w(.0) { ; }

};

/*
 * functions provided -- feel free to use
 *
 * loadMap: reads in a number of walls from a map file
 *          created in the Nomad simulator
 */

void
loadMap(vector<Wall>& w, char* mapfile)
{
  ifstream map(mapfile);

  int nvertices;
  int i=0;

  while (map >> nvertices)
  {
    if (nvertices != 4) {
      double notused;
      // take in unused walls
      for (int j=0 ; j<nvertices*2 ; ++j) cin >> notused;
      continue; // only handles rectangles now
    }

    double llx, lly, lrx, lry, ulx, uly, urx, ury;

    map >> llx >> lly >> lrx >> lry >> urx >> ury >> ulx >> uly;

    w.push_back(Wall(Wall::H,0,lly,llx,lrx)); // bottom
    w.push_back(Wall(Wall::V,lrx,0,lry,ury)); // right
    w.push_back(Wall(Wall::H,0,uly,ulx,urx)); // top
    w.push_back(Wall(Wall::V,llx,0,lly,uly)); // left
  }
}

/*
 * drawParticles: displays the individual pose guesses
 *                in the simulator
 */

void
drawParticles(vector<Particle>& pFilt, int radius, int color)
{
  int NPART = pFilt.size();

  for (int i=0 ; i<NPART ; ++i)
  {
    double x = pFilt[i].x;
    double y = pFilt[i].y;
    double w = pFilt[i].w;
    if (w > 0.0) {          // don't draw the particle if the weight is 0.0
      draw_arc(x-(radius/2),y+(radius/2),radius,radius,0,3600,color);
    }
  }
}

/*
 * moveParticles: moves each particle in a particle filter
 *                by dx and dy
 *
 *                It also adds some Gaussian noise
 *                with a stdeviation of POS_STDEV_FRAC
 *                of the original move, i.e., the error
 *                is proportional to the size of the move.
 */

void
moveParticles(vector<Particle>& pFilt, double dx, double dy)
{
  int NPART = pFilt.size();

  for (int i=0 ; i<NPART ; ++i)
  {
    pFilt[i].x += dx + POS_STDEV_FRAC*dx*gaussrand();
    pFilt[i].y += dy + POS_STDEV_FRAC*dy*gaussrand();
  }
}

/*
 * clearance: determine the distance to the nearest
 *            wall in the specified direction
 */

double
clearance(vector<Wall>& w, double x, double y, int dir)
{
  double result = 1.0e9;
  double diff = 0.0;

  switch (dir) {
    case POSITIVE_Y:
      {
        for (int i=0 ; i<w.size() ; ++i)
        {
          // look only for horizontal lines with 
          // y-coordinates above y and
          // low and high surrounding x
          if (w[i].dir == Wall::H && w[i].y >= y && 
              w[i].low <= x && w[i].high >= x)
          {
            diff = w[i].y - y;
            if (diff < result) result = diff;
          }
        }
      }
      break;
    case POSITIVE_X:
      {
        for (int i=0 ; i<w.size() ; ++i)
        {
          if (w[i].dir == Wall::V && w[i].x >= x && 
              w[i].low <= y && w[i].high >= y)
          {
            diff = w[i].x - x;
            if (diff < result) result = diff;
          }
        }
      }
      break;
    case NEGATIVE_Y:
      {
        for (int i=0 ; i<w.size() ; ++i)
        {
          if (w[i].dir == Wall::H && w[i].y <= y && 
              w[i].low <= x && w[i].high >= x)
          {
            diff = y - w[i].y;
            if (diff < result) result = diff;
          }
        }
      }
      break;
    case NEGATIVE_X:
      {
        for (int i=0 ; i<w.size() ; ++i)
        {
          if (w[i].dir == Wall::V && w[i].x <= x && 
              w[i].low <= y && w[i].high >= y)
          {
            diff = x - w[i].x;
            if (diff < result) result = diff;
          }
        }
      }
      break;
    default:
      break;
  }

  return result;
}

/*
 * gaussianPDF: return the (relative) probability that x
 *              was drawn from a Normal distribution with
 *              mean m and standard deviation stdev
 */

double
gaussianPDF(double x, double m, double stdev)
{
  // we don't bother with the constant in front, as everything will
  // get normalized anyway...
  //
  // just for the record, the constant would be
  // 1/(stdev*sqrt(2.0*pi))

  double diff = x-m;
  double exponent = -1.0*diff*diff/(2.0*stdev*stdev);

  return exp(exponent);
}


/*
 * gaussrand: A function to generate a normal random variable 
 *            with mean 0 and standard deviation of 1.  
 *
 * To adjust this normal distribution, multiply by the new standard
 * deviation and add the mean.  This uses the so-called Box-Muller method.
 *
 * note: drand48() is a function that returns a uniformly distributed random
 * number from 0.0 to 1.0
 *
 * from: http://remus.rutgers.edu/~rhoads/Code/code.html
 */

double 
gaussrand()
{
  static double V2, fac;
  static int phase = 0;
  double S, Z, U1, U2, V1;

  if (phase)
    Z = V2 * fac;
  else
  {
    do {
      U1 = drand48();
      U2 = drand48();

      V1 = 2 * U1 - 1;
      V2 = 2 * U2 - 1;
      S = V1 * V1 + V2 * V2;
    } while(S >= 1);

    fac = sqrt (-2 * log(S) / S);
    Z = V1 * fac;
  }

  phase = 1 - phase;

  return Z;
}


/*
 * True if particle is past a wall.
 */
bool out_of_bounds(vector<Wall>& walls, Particle p)
{
    if (clearance(walls, p.x, p.y, NEGATIVE_X) <= 0
	|| clearance(walls, p.x, p.y, POSITIVE_X) <= 0
	|| clearance(walls, p.x, p.y, NEGATIVE_Y) <= 0
	|| clearance(walls, p.x, p.y, POSITIVE_Y) <= 0)
	return true;
    else
	return false;
}

/*
 * Normalize modifies probabilities such that they sum to 1.
 */
void normalize(vector<Particle>& pFilt)
{
    double sum = 0.0;
    for (int i = 0; i < NPART; i++)
    {
	sum += (double)pFilt[i].w;
    }
    for (int i = 0; i < NPART; i++)
    {
	pFilt[i].w = (double)pFilt[i].w / sum;
    }
}


/*
 * Resample divides heavily weighted particles into multiple particles
 * which will each have a smaller weight, and gets rid of particles with
 * a very low weight.
 */
void resample(vector<Particle>& pFilt)
{
    vector<Particle> newArray(NPART);

    for (int i = 0; i < NPART; i++)
	newArray[i] = Particle();

    double prob = 0.0;

    double cut = (double)1.0 / (NPART + 1);

    int j = 0;

    /*
     * We scan through the old particles to create the new ones.
     */
    for (int i = 0; i < NPART; i++)
    {
	prob += pFilt[i].w;

	while (prob > cut)
	{
	    newArray[j].x = (double)NOISE_MODEL * (gaussrand() - 0.5) 
		+ pFilt[i].x;
	    newArray[j].y = (double)NOISE_MODEL * (gaussrand() - 0.5) 
		+ pFilt[i].y;
	    newArray[j].w = (double) 1.0 / NPART;
	    j++;
	    cut += (double)1.0 / (NPART + 1);
	}
    }

    pFilt = newArray;

}



/****************************
 *
 *  The usual main method
 *
 ****************************/

int
main(int argc, char** argv)
{

  SERV_TCP_PORT = 7021;    // Be sure to alter this to match your server

  /*
   * initialize random number generator
   *   you can choose your own long int here to test repeatably
   */

  long int L = T.randomLongInt();
  srand48(L);

  /*
   * lots of variables that will be useful
   */

  double oldx = 0, x = 0;
  double oldy = 0, y = 0;
  double dx = 0, dy = 0;

  double north_sonar, west_sonar, south_sonar, east_sonar;

  /*
   * read in the file of parameters
   */

  char comment[100];
  ifstream inf("./parameters");
  inf >> COLOR >> comment;
  cout << "COLOR is " << COLOR << endl;
  inf >> COLOR2 >> comment;
  cout << "COLOR2 is " << COLOR2 << endl;
  inf >> PARTICLE_RADIUS  >> comment;
  cout << "PARTICLE_RADIUS is " << PARTICLE_RADIUS << endl;
  inf >> NOISE >> comment;
  cout << "NOISE is " << NOISE << endl;
  inf >> NOISE_MODEL >> comment;
  cout << "NOISE_MODEL is " << NOISE_MODEL << endl;
  inf >> NPART >> comment;
  cout << "NPART is " << NPART << endl;
  inf >> POS_STDEV_FRAC >> comment;
  cout << "POS_STDEV_FRAC is " << POS_STDEV_FRAC << endl;

  if (argc > 2)     /* for command-line parameters */
  {
    /*  // not used at the moment
    Q = atof(argv[1]);
    cout << "Setting Q to " << Q << endl;
    R = atof(argv[2]);
    cout << "Setting R to " << R << endl;
    */
  }

  /*
   * get the environment
   */

  vector<Wall> walls; 
  loadMap(walls, "./mclmap1");
  cout << "walls.size is " << walls.size() << endl;

  /*
   *  debugging statement
   *
  for (int i=0 ; i<walls.size() ; ++i)
  {
    cout << "Wall #" << i << endl;
    cout << "  dir: " << ( walls[i].dir==Wall::V ? "VERT" : "HORIZ" ) << endl;
    cout << "    x: " << walls[i].x << endl;
    cout << "    y: " << walls[i].y << endl;
    cout << "  low: " << walls[i].low << endl;
    cout << " high: " << walls[i].high << endl;
    cout << endl;
  }
  */

  x = 0; y = 0;
  double ycl = clearance(walls,x,y,POSITIVE_Y);
  double xcl = clearance(walls,x,y,POSITIVE_X);
  double nycl = clearance(walls,x,y,NEGATIVE_Y);
  double nxcl = clearance(walls,x,y,NEGATIVE_X);
  cout << "+ycl is " << ycl << endl;
  cout << "+xcl is " << xcl << endl;
  cout << "-ycl is " << nycl << endl;
  cout << "-xcl is " << nxcl << endl;
  

  /* 
   * connect to the simulator
   */

  if (connect_robot(1)) 
  {
    printf("Connected to robot.\n");
    printf("Hit q to exit.\n");
  }
  else 
  {
    printf("Failed to connect to robot\n");
    exit(0);
  }

  /* 
   * zero the robot
   */

  zr();

  /* 
   * turn on all sensors 
   */

  for (int i=1 ; i<=42 ; ++i)  Smask[i] = 1;
  ct();

  gs();                       /* update the robot's state           */
  refresh_all();              /* clear the graphics                 */

  /*
   * the KbdInput object allows nonblocking input
   */

  KbdInput KI; 
  char c; 

  /*
   * the sensing loop
   */

  gs();                       /* update state                        */


  /*
   * initial set of particles
   */

  vector<Particle> pFilt(NPART);
  for (int i=0 ; i<NPART ; ++i)
  {
    // this assumes the robot's environment
    // is between -2000 and 2000 in the x and y dimensions
    double x = (drand48()-0.5)*2*2000;
    double y = (drand48()-0.5)*2*2000;
    double w = 1.0/NPART;              // equal weights
    pFilt[i] = Particle(x,y,w);
  }

  drawParticles(pFilt, PARTICLE_RADIUS, COLOR);

  /*
   * main loop
   */

  bool localizing = true;
  while (localizing)
  {



    /*
     * move the robot with the keyboard
     */

    KI.off();

    cout << "Move the robot to a new location, using\n"
         << "   e   (forward)\n"
         << "   d   (backward)\n"
         << "   s   (turn left)\n"
         << "   f   (turn right)\n\n"
         << "   q   (to stop moving)\n\n"
         << "   Q   (to quit the program)\n\n";

    KI.on();                    /* turn on raw input/output            */

    while (1) 
    {
      gs();
      c = KI.nextChar();      /* c is 0 if there's no input ready     */

      if ( c == 'q' )    // q quits moving the robot
      {
	vm(0, 0, 0);
	break;
      }

      if ( c == 'Q' )    // quits for good
      {
        KI.off();                 /* reset terminal settings              */
        disconnect_robot(1);      /* gracefully disconnect from the robot */
        exit(0);
      }

      if ( c == 'f' )    // turn right
      {
        vm(0,-100,0); 
      }

      if ( c == 'e' )    // go forward
      {
        vm(100,0,0); 
      }
      if ( c == 'd' )    // go backwards
      {
        vm(-100,0,0); 
      }

      if ( c == 's' )    // turn left
      {
        vm(0,100,0); 
      }

      if ( c == ' ' )    // stop
      {
        vm(0,0,0);
      }
    }

    KI.off();

    /*
     * now the robot's move has ended
     * 
     * we obtain the actual location and newly achieved offset
     * of the robot with the following
     */

    gs();                       /* update the robot's state           */
    oldx = x;                   
    oldy = y;
    x = State[34];              // actual x position
    y = State[35];              // actual y position

    dx = x-oldx;
    dy = y-oldy;

    /*
     * move the particles appropriately and redraw them
     *   
     *   NOTE #1: the noise in the robot's motion model
     *        is simulated in the "moveParticles" function
     *        The parameter POS_STDEV_FRAC is the fraction
     *        of the move that will be used as one standard
     *        for the possible end locations for each particle
     *
     *   NOTE #2: the noise needs to be added per-particle;
     *        otherwise, all the particles would move with
     *        exactly the same displacement!
     */

    moveParticles(pFilt, dx, dy);
    drawParticles(pFilt,PARTICLE_RADIUS,COLOR2);

    /*
     * get sonar readings (with added noise, of course!)
     *   the units are tenths of an inch, which are the units
     *   used in the map (the Walls) and the simulator display
     *
     *   NOISE is one standard deviation of the spread of the
     *         sensor noise
     *
     *   The 90 is the radius of the robot (in tenths of an inch)
     */

    north_sonar = 90 + 10*State[21] + NOISE*gaussrand();
    west_sonar  = 90 + 10*State[25] + NOISE*gaussrand();
    south_sonar = 90 + 10*State[29] + NOISE*gaussrand();
    east_sonar  = 90 + 10*State[17] + NOISE*gaussrand();

    /*
     * update the weight of each particle
     *
     *   to do this, you'll need to compare the noisy sonar
     *   readings (set above) with the values that each
     *   particle expects (you might want to use the provided
     *   "clearance" function to find the expected distance to
     *   obstacles)
     *
     *   HINT #1: use a gaussian model (with the gaussianPDF
     *            function) with NOISE_MODEL as the st. deviation
     *
     *   HINT #2: you may want to choose only some of the 4 sonar
     *            readings to use here, at least at first
     *
     *  then, you'll want to normalize the particle filter
     *  and resample it before the next iteration
     *
     *  you should draw the results to the simulator after 
     *  each of these steps, at least at first -- perhaps with
     *  the radius depending on each particle's weight
     *
     *  to clear the smiulator's graphics (it can get cluttered
     *  if drawing lots of particles many times), use
     *
     *    refresh_all();
     */

    for (int i = 0; i < NPART; i++)
    {
	/*
	 * First, we clear out the particles that have moved through obstacles.
	 */

	if (out_of_bounds(walls, pFilt[i]))
	{
	    pFilt[i].w = 0;
	}
	
	else
	{
	    double neg_x;
	    double pos_x;
	    double neg_y;
	    double pos_y;

	    neg_x = clearance(walls, pFilt[i].x, pFilt[i].y, NEGATIVE_X);
	    pos_x = clearance(walls, pFilt[i].x, pFilt[i].y, POSITIVE_X);
	    neg_y = clearance(walls, pFilt[i].x, pFilt[i].y, NEGATIVE_Y);
	    pos_y = clearance(walls, pFilt[i].x, pFilt[i].y, POSITIVE_Y);

	    /*
	     * We need to ignore maxed out sonar readings if the clearance
	     * is also large.
	     */

	    if (west_sonar >= 2550 && neg_x >= 2550)
	    {
		neg_x = 2550.0;
	    }
	    if (east_sonar >= 2550 && pos_x >= 2550)
	    {
		pos_x = 2550.0;
	    }
	    if (north_sonar >= 2550 && pos_y >= 2550)
	    {
		pos_y = 2550.0;
	    }
	    if (south_sonar >= 2550 && neg_y >= 2550)
	    {
		neg_y = 2550.0;
	    }

	    pFilt[i].w = 1.0;

	    if (west_sonar < 2550)
	    {
		pFilt[i].w *= gaussianPDF(neg_x, west_sonar, NOISE_MODEL);
	    }
	    if (east_sonar < 2550)
	    {
		pFilt[i].w *= gaussianPDF(pos_x, east_sonar, NOISE_MODEL);
	    }
	    if (south_sonar < 2550)
	    {
		pFilt[i].w *= gaussianPDF(neg_y, south_sonar, NOISE_MODEL);
	    }
	    if (north_sonar < 2550)
	    {
		pFilt[i].w *= gaussianPDF(pos_y, north_sonar, NOISE_MODEL);
	    }
	}
    }


    /*
     * Normalize, resample, redraw.
     */

    normalize(pFilt);

    resample(pFilt);
    
    refresh_all();
    drawParticles(pFilt, PARTICLE_RADIUS, COLOR);


  }  // end of while (localizing) loop



  KI.off();                 /* reset terminal settings              */

  disconnect_robot(1);      /* gracefully disconnect from the robot */


}



